% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/import_isd_hourly.R
\name{import_isd_hourly}
\alias{import_isd_hourly}
\title{Import Meteorological data from the NOAA Integrated Surface Database (ISD)}
\usage{
import_isd_hourly(
  code = "037720-99999",
  year = 2014,
  hourly = TRUE,
  source = c("delim", "fwf"),
  progress = rlang::is_interactive(),
  quiet = FALSE
)
}
\arguments{
\item{code}{The identifying code as a character string. The code is a
combination of the USAF and the WBAN unique identifiers. The codes are
separated by a \dQuote{-} e.g. \code{code = "037720-99999"}.}

\item{year}{The year to import. This can be a vector of years e.g. \code{year = 2000:2005}.}

\item{hourly}{Should hourly means be calculated? The default is \code{TRUE}. If
\code{FALSE} then the raw data are returned.}

\item{source}{The NOAA ISD service stores files in two formats; as delimited
CSV files (\code{"delim"}) and as fixed width files (\code{"fwf"}).
\code{\link[=import_isd_hourly]{import_isd_hourly()}} defaults to \code{"delim"} but, if the delimited data
store is down, users may wish to try \code{"fwf"} instead. Both data sources
should be identical to one another.}

\item{progress}{Show a progress bar when importing many stations/years?
Defaults to \code{TRUE} in interactive R sessions. Passed to \code{.progress} in
\code{\link[purrr:map]{purrr::map()}} and/or \code{\link[purrr:pmap]{purrr::pmap()}}.}

\item{quiet}{Print missing sites/years to the screen? Defaults to \code{FALSE}.}
}
\value{
Returns a data frame of surface observations. The data frame is
consistent for use with the \code{openair} package. Note that the data are
returned in GMT (UTC) time zone format. Users may wish to express the data
in other time zones, e.g., to merge with air pollution data.
}
\description{
This is the main function to import data from the NOAA Integrated Surface
Database (ISD). The ISD contains detailed surface meteorological data from
around the world for over 30,000 locations.
}
\details{
Note the following units for the main variables:

\describe{

\item{date}{Date/time in POSIXct format. \strong{Note the time zone is GMT (UTC)
and may need to be adjusted to merge with other local data. See details
below.}}

\item{latitude}{Latitude in decimal degrees (-90 to 90).}

\item{longitude}{Longitude in decimal degrees (-180 to 180). Negative numbers
are west of the Greenwich Meridian.}

\item{elevation}{Elevation of site in metres.}

\item{wd}{Wind direction in degrees. 90 is from the east.}

\item{ws}{Wind speed in m/s.}

\item{ceil_hgt}{The height above ground level (AGL) of the lowest cloud or
obscuring phenomena layer aloft with 5/8 or more summation total sky cover,
which may be predominantly opaque, or the vertical visibility into a
surface-based obstruction.}

\item{visibility}{The visibility in metres.}

\item{air_temp}{Air temperature in degrees Celcius.}

\item{dew_point}{The dew point temperature in degrees Celcius.}

\item{atmos_pres}{The sea level pressure in millibars.}

\item{RH}{The relative humidity (\%).}

\item{cl_1,  ...,  cl_3}{Cloud cover for different layers in Oktas (1-8).}

\item{cl}{Maximum of cl_1 to cl_3 cloud cover in Oktas (1-8).}

\item{cl_1_height, ..., cl_3_height}{Height of the cloud base for each later
in metres.}

\item{precip_12}{12-hour precipitation in mm. The sum of this column should
give the annual precipitation.}

\item{precip_6}{6-hour precipitation in mm.}

\item{precip}{This value of precipitation spreads the 12-hour total across
the previous 12 hours.}

\item{pwc}{The description of the present weather description (if
available).}

}

The data are returned in GMT (UTC). It may be necessary to adjust the time
zone when combining with other data. For example, if air quality data were
available for Beijing with time zone set to "Etc/GMT-8" (note the negative
offset even though Beijing is ahead of GMT. See the \code{openair} package and
manual for more details), then the time zone of the met data can be changed
to be the same. One way of doing this would be \code{attr(met$date, "tzone") <- "Etc/GMT-8"} for a meteorological data frame called \code{met}. The two data sets
could then be merged based on \code{date}.
}
\section{Parallel Processing}{


If you are importing a lot of meteorological data, this can take a long
while. This is because each combination of year and station requires
downloading a separate data file from NOAA's online data directory, and the
time each download takes can quickly add up. Many data import functions in
\code{{worldmet}} can use parallel processing to speed downloading up, powered
by the capable \code{{mirai}} package. If users have any \code{{mirai}} "daemons"
set, these functions will download files in parallel. The greatest benefits
will be seen if you spawn as many daemons as you have cores on your
machine, although one fewer than the available cores is often a good rule
of thumb. Your mileage may vary, however, and naturally spawning more
daemons than station-year combinations will lead to diminishing returns.

\if{html}{\out{<div class="sourceCode">}}\preformatted{# set workers - once per session
mirai::daemons(4)

# import lots of data - NB: no change in the import function!
big_met <- import_ghcn_hourly(code = "UKI0000EGLL", year = 2010:2025)
}\if{html}{\out{</div>}}
}

\examples{

\dontrun{
# import some data
beijing_met <- import_isd_hourly(code = "545110-99999", year = 2010:2011)

# importing lots of data? use mirai for parallel processing
mirai::daemons(4)
beijing_met2 <- import_isd_hourly(code = "545110-99999", year = 2010:2025)
}
}
\seealso{
Other NOAA ISD functions: 
\code{\link{import_isd_lite}()},
\code{\link{import_isd_stations}()},
\code{\link{import_isd_stations_live}()}
}
\author{
David Carslaw
}
\concept{NOAA ISD functions}
