% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsesimp.R
\name{tsesimp}
\alias{tsesimp}
\title{Simple Two-Stage Estimation (TSEsimp) for Treatment Switching}
\usage{
tsesimp(
  data,
  id = "id",
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  censor_time = "censor_time",
  pd = "pd",
  pd_time = "pd_time",
  swtrt = "swtrt",
  swtrt_time = "swtrt_time",
  base_cov = "",
  base2_cov = "",
  aft_dist = "weibull",
  strata_main_effect_only = TRUE,
  recensor = TRUE,
  admin_recensor_only = TRUE,
  swtrt_control_only = TRUE,
  alpha = 0.05,
  ties = "efron",
  offset = 1,
  boot = TRUE,
  n_boot = 1000,
  seed = 0,
  nthreads = 0
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{id}: The subject id.
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{pd}: The disease progression indicator, 1=PD, 0=no PD.
\item \code{pd_time}: The time from randomization to disease progression.
\item \code{swtrt}: The treatment switch indicator, 1=switch, 0=no switch.
\item \code{swtrt_time}: The time from randomization to treatment switch.
\item \code{base_cov}: The baseline covariates (excluding treat).
\item \code{base2_cov}: The baseline and secondary baseline
covariates (excluding treat).
}}

\item{id}{The name of the id variable in the input data.}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{pd}{The name of the pd variable in the input data.}

\item{pd_time}{The name of the pd_time variable in the input data.}

\item{swtrt}{The name of the swtrt variable in the input data.}

\item{swtrt_time}{The name of the swtrt_time variable in the input data.}

\item{base_cov}{The names of baseline covariates (excluding
treat) in the input data for the outcome Cox model.}

\item{base2_cov}{The names of baseline and secondary baseline covariates
(excluding treat) in the input data for the AFT model for
post-progression survival.}

\item{aft_dist}{The assumed distribution for time to event for the AFT
model. Options include "exponential", "weibull" (default),
"loglogistic", and "lognormal".}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the AFT model. Defaults to \code{TRUE}, otherwise all
possible strata combinations will be considered in the AFT model.}

\item{recensor}{Whether to apply recensoring to counterfactual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring times only. Defaults to \code{TRUE}. If \code{FALSE},
recensoring will be applied to the actual censoring times for dropouts.}

\item{swtrt_control_only}{Whether treatment switching occurred only in
the control group. The default is \code{TRUE}.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{offset}{The offset to calculate the time disease progression to
death or censoring. We can set \code{offset} equal to 0 (no offset),
and 1 (default), 1/30.4375, or 1/365.25 if the time unit is day, month,
or year, respectively.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{TRUE}.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results.}

\item{nthreads}{The number of threads to use in bootstrapping (0 means
the default RcppParallel behavior)}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter for the control group.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "AFT model" or "bootstrap".
\item \code{pvalue}: The two-sided p-value.
\item \code{pvalue_type}: The type of two-sided p-value for treatment effect,
i.e., "Cox model" or "bootstrap".
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "Cox model" or "bootstrap".
\item \code{event_summary}: A data frame containing the count and percentage
of deaths, disease progressions, and switches by treatment arm.
\item \code{data_aft}: A list of input data for the AFT model by treatment
group. The variables include \code{id}, \code{stratum}, \code{"pps"},
\code{"event"}, \code{"swtrt"}, \code{base2_cov}, \code{pd_time},
\code{swtrt_time}, and \code{time}.
\item \code{fit_aft}: A list of fitted AFT models by treatment group.
\item \code{res_aft}: A list of deviance residuals from the AFT models
by treatment group.
\item \code{data_outcome}: The input data for the outcome Cox model
of counterfactual unswitched survival times.
The variables include \code{id}, \code{stratum}, \code{"t_star"},
\code{"d_star"}, \code{"treated"}, \code{base_cov}, and \code{treat}.
\item \code{km_outcome}: The Kaplan-Meier estimates of the survival
functions for the treatment and control groups based on the
counterfactual unswitched survival times.
\item \code{lr_outcome}: The log-rank test results for the treatment
effect based on the counterfactual unswitched survival times.
\item \code{fit_outcome}: The fitted outcome Cox model.
\item \code{fail}: Whether a model fails to converge.
\item \code{psimissing}: Whether the \code{psi} parameter cannot be estimated.
\item \code{settings}: A list containing the input parameter values.
\item \code{psi_trt}: The estimated causal parameter for the experimental
group if \code{swtrt_control_only} is \code{FALSE}.
\item \code{psi_trt_CI}: The confidence interval for \code{psi_trt}
if \code{swtrt_control_only} is \code{FALSE}.
\item \code{fail_boots}: The indicators for failed bootstrap samples
if \code{boot} is \code{TRUE}.
\item \code{fail_boots_data}: The data for failed bootstrap samples
if \code{boot} is \code{TRUE}.
\item \code{hr_boots}: The bootstrap hazard ratio estimates
if \code{boot} is \code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates
if \code{boot} is \code{TRUE}.
\item \code{psi_trt_boots}: The bootstrap \code{psi_trt} estimates
if \code{boot} is \code{TRUE} and \code{swtrt_control_only} is
\code{FALSE}.
}
}
\description{
Estimates the causal parameter by fitting an accelerated
failure time (AFT) model comparing post-progression survival between
switchers and non-switchers, and derives the adjusted hazard ratio
from the Cox model using counterfactual \emph{unswitched} survival
times based on the estimated causal parameter.
}
\details{
Assuming one-way switching from control to treatment, the
hazard ratio and confidence interval under a no-switching scenario
are obtained as follows:
\itemize{
\item Estimate the causal parameter \eqn{\psi} by fitting an AFT model
comparing post-progression survival between switchers and non-switchers
in the control group who experienced disease progression.
\item Compute counterfactual survival times for control patients using
the estimated \eqn{\psi}.
\item Fit a Cox model to the observed survival times for the treatment group
and the counterfactual survival times for the control group to
estimate the hazard ratio.
\item When bootstrapping is used, derive the confidence interval and
p-value for the hazard ratio from a t-distribution with
\code{n_boot - 1} degrees of freedom.
}

If treatment switching occurs before or in the absence of recorded disease
progression, the patient is considered to have progressed at the time of
treatment switching.
}
\examples{

library(dplyr)

# modify pd and dpd based on co and dco
shilong <- shilong \%>\%
  mutate(dpd = ifelse(co & !pd, dco, dpd),
         pd = ifelse(co & !pd, 1, pd)) \%>\%
  mutate(dpd = ifelse(pd & co & dco < dpd, dco, dpd))
  
# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  slice(n()) \%>\%
  select(-c("ps", "ttc", "tran"))

# the last value of time-dependent covariates before pd
shilong2 <- shilong \%>\%
  filter(pd == 0 | tstart <= dpd) \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  slice(n()) \%>\%
  select(bras.f, id, ps, ttc, tran)

# combine baseline and time-dependent covariates
shilong3 <- shilong1 \%>\%
  left_join(shilong2, by = c("bras.f", "id"))

# apply the two-stage method
fit1 <- tsesimp(
  data = shilong3, id = "id", time = "tstop", event = "event",
  treat = "bras.f", censor_time = "dcut", pd = "pd",
  pd_time = "dpd", swtrt = "co", swtrt_time = "dco",
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                "pathway.f"),
  base2_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                "pathway.f", "ps", "ttc", "tran"),
  aft_dist = "weibull", alpha = 0.05,
  recensor = TRUE, swtrt_control_only = FALSE, offset = 1,
  boot = FALSE)

fit1

}
\references{
Nicholas R Latimer, KR Abrams, PC Lambert, MK Crowther, AJ Wailoo,
JP Morden, RL Akehurst, and MJ Campbell.
Adjusting for treatment switching in randomised controlled trials - A
simulation study and a simplified two-stage method.
Statistical Methods in Medical Research. 2017;26(2):724-751.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
