% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map_discrete_to_vad.R
\name{map_discrete_to_vad}
\alias{map_discrete_to_vad}
\title{Map Discrete Emotions to VAD (Valence-Arousal-Dominance) Framework}
\usage{
map_discrete_to_vad(
  results,
  mapping = "nrc_vad",
  weighted = TRUE,
  cache_lexicon = TRUE,
  vad_lexicon = NULL
)
}
\arguments{
\item{results}{Output from image_scores(), transformer_scores(), or video_scores().
Can be a data.frame (from image/video functions) or a list (from transformer functions).}

\item{mapping}{Character. Which VAD mapping to use. Currently supports:
\itemize{
  \item \code{"nrc_vad"}: Uses NRC VAD lexicon (Mohammad, 2018)
}}

\item{weighted}{Logical. Whether to compute weighted averages based on confidence
scores (default: TRUE). If FALSE, performs simple lookup of the highest-scoring emotion.}

\item{cache_lexicon}{Logical. Whether to cache the VAD lexicon for repeated use
(default: TRUE).}

\item{vad_lexicon}{Optional data.frame. Pre-loaded VAD lexicon data to use instead
of loading from textdata. Must have columns for word, valence, arousal, dominance
(accepts both lowercase and capitalized versions, e.g., Word/word, Valence/valence).
If provided, the function will use this data directly.}
}
\value{
A data.frame with columns:
\itemize{
  \item \code{valence}: Valence score (positive vs negative emotion)
  \item \code{arousal}: Arousal score (excitement vs calmness)
  \item \code{dominance}: Dominance score (control vs submissiveness)
}
For transformer_scores() input, includes additional identifier columns.
For image/video_scores() input, returns one row per input row.
}
\description{
Maps discrete emotion classifications from image_scores(), transformer_scores(),
or video_scores() functions to the Valence-Arousal-Dominance (VAD) framework
using published lexicons. Automatically downloads the NRC VAD lexicon via
textdata package on first use.
}
\details{
This function bridges discrete emotion classification outputs with the continuous
VAD emotion framework. The VAD model represents emotions in a three-dimensional
space where:
\itemize{
  \item **Valence**: Pleasantness (positive/negative)
  \item **Arousal**: Activation level (excited/calm)
  \item **Dominance**: Control (dominant/submissive)
}

**Input Type Detection:**
The function automatically detects the input type:
\itemize{
  \item **data.frame**: Assumes output from image_scores() or video_scores()
  \item **list**: Assumes output from transformer_scores()
}

**Weighting Methods:**
\itemize{
  \item **weighted = TRUE**: Computes weighted average VAD scores based on
    classification confidence scores
  \item **weighted = FALSE**: Uses VAD values for the highest-scoring emotion only
}

**VAD Mappings:**
Currently supports the NRC VAD lexicon which provides VAD ratings for emotion
words based on crowdsourced annotations. The lexicon must be downloaded first
using `textdata::lexicon_nrc_vad()` in an interactive session.

**Setup Required:**
Before using this function, download the NRC VAD lexicon by running:
`textdata::lexicon_nrc_vad()` in an interactive R session and accepting the license.
}
\section{Data Privacy}{

  VAD lexicon is downloaded once and cached locally. No data is sent to external servers.
}

\examples{
\dontrun{
# Method 1: Auto-load from textdata (requires prior download)
textdata::lexicon_nrc_vad()  # Run once to download

# With image scores
image_path <- system.file("extdata", "boris-1.png", package = "transforEmotion")
emotions <- c("joy", "sadness", "anger", "fear", "surprise", "disgust")
img_results <- image_scores(image_path, emotions)
vad_results <- map_discrete_to_vad(img_results)

# Method 2: Download once and pass as argument (recommended)
nrc_vad <- textdata::lexicon_nrc_vad()  # Download once

# Use with different emotion results
vad_results1 <- map_discrete_to_vad(img_results, vad_lexicon = nrc_vad)

text <- "I am so happy today!"
trans_results <- transformer_scores(text, emotions)
vad_results2 <- map_discrete_to_vad(trans_results, vad_lexicon = nrc_vad)

# Simple lookup (no weighting)
vad_simple <- map_discrete_to_vad(img_results, weighted = FALSE, vad_lexicon = nrc_vad)
}

}
\references{
Mohammad, S. M. (2018). Obtaining reliable human ratings of valence, arousal, and
dominance for 20,000 English words. Proceedings of the 56th Annual Meeting of the
Association for Computational Linguistics (Volume 1: Long Papers), 174-184.
}
\author{
Aleksandar Tomasevic <atomashevic@gmail.com>
}
