% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hfdr_class.R
\docType{class}
\name{hypothesesTree-class}
\alias{hypothesesTree-class}
\alias{initialize,hypothesesTree-method}
\alias{show,hypothesesTree-method}
\alias{summary,hypothesesTree-method}
\alias{plot,hypothesesTree,ANY-method}
\title{Manage Hierarchical Test Results}
\usage{
\S4method{initialize}{hypothesesTree}(.Object, ...)

\S4method{show}{hypothesesTree}(object)

\S4method{summary}{hypothesesTree}(object)

\S4method{plot}{hypothesesTree,ANY}(
  x,
  ...,
  adjust = TRUE,
  return_script = FALSE,
  width = 900,
  height = 500,
  base_font_size = 12,
  output_file_name = paste("hyp_tree", gsub("[^\\\\d]+", "", Sys.time(), perl = TRUE),
    ".html", sep = "")
)
}
\arguments{
\item{.Object}{Dummy to initialize S4 class}

\item{...}{Any other arguments are accepted, but they will be ignored.}

\item{object}{A hypothesesTree object whose hypotheses we want to summarize.}

\item{x}{A hypothesesTree object whose hypotheses we want to plot.}

\item{adjust}{Show the adjusted p-values?}

\item{return_script}{Return the d3 code used to generate the plot.}

\item{width}{The width of the printed tree, in pixels.}

\item{height}{The height of the printed tree, in pixels.}

\item{base_font_size}{The size of the plot's labels.}

\item{output_file_name}{The name of the file to which the script is saved.}
}
\description{
This defines a class of hypothesis trees. It should allow us to manipulate
the hypotheses, the results of tests associated with them, and their
hierarchical relation. We set a class to make this manpiulation easier and to
prevent the bugs that arise from complexity. Note that the tree is defined by
its adjacency matrix.

Check that the hypotheses tree is correctly initialized. It ensures that the
number of unadjusted p-values, hypotheses names, and nodes in the hypotheses
tree all agree. It also checks that the hypotheses tree is in fact a tree.

This prints the unadjusted and adjusted p-values of the hypotheses tree
associated with the HFDR procedure.

This prints the most significant adjusted p-values, along with
  estimates of the FDR across the tree and at tips.

Creates an interactive plot to understand the tests along the tree that were
rejected, with their adjusted and unadjusted p-values.
}
\section{Slots}{

\describe{
\item{\code{tree}}{Object of class \code{"matrix"}. The edgelist for the hypotheses
tree.
  * hypothesisIndex: The index of the current hypothesis in the
     \code{unadjp} vector
  * hypothesisName: The name of the current hypothesis, from the names of
     the \code{unadjp} vector
  * unadjp: The unadjusted p-values input from \code{unadjp}
  * adjp: The adjusted p-values, after the GBH adjustment.
  * group: The group to which the original hypothesis belonged
  * significance: A code for the significance of each hypothesis}

\item{\code{p.vals}}{Object of class \code{'data.frame'}. Each row corresponds to an
individual hypothesis. The first column stores the p-values before GBH
adjustment, while the second gives the hFDR adjusted p-values. The
hypotheses are sorted in order of significance according to these GBH
adjusted p-values. The \code{group} column gives the group membership of
each hypothesis, and \code{adj.significnace} codes the significance of each
hypothesis, according to the GBH adjusted p-values.}

\item{\code{alpha}}{Object of class \code{"numeric"}. The level at which the FDR is
controlled among children of each parent node.}
}}

\examples{
library('igraph')
library('ape')

alternative.indices <- sample(1:49, 30)
unadj.p.values <- vector("numeric", length = 49)
unadj.p.values[alternative.indices] <- runif(30, 0, 0.01)
unadj.p.values[-alternative.indices] <- runif(19, 0, 1)
unadj.p.values[c(1:5)] <- runif(5, 0, 0.01)
names(unadj.p.values) <- paste("Hyp ", c(1:49))

tree <- as.igraph(rtree(25))
V(tree)$name <- names(unadj.p.values)
tree.el <- get.edgelist(tree)

hyp.tree <- hFDR.adjust(unadj.p.values, tree.el, 0.05)
if (interactive()) {
  plot(hyp.tree)
}
}
