test_that("is_colour identifies valid hex colours", {
  # Valid hex colours
  expect_true(unname(is_colour("#000000")))
  expect_true(unname(is_colour("#ffffff")))
  expect_true(unname(is_colour("#ff00ff")))
  expect_true(unname(is_colour("#010101")))

  # Valid colour names
  expect_true(unname(is_colour("red")))
  expect_true(unname(is_colour("blue")))
  expect_true(unname(is_colour("white")))

  # Vector of valid colours
  valid_colors <- c("#ff00ff", "#010101", "red", "blue")
  expect_equal(unname(is_colour(valid_colors)), rep(TRUE, 4))
})

test_that("is_colour identifies invalid colours", {
  # Invalid hex colours
  expect_false(unname(is_colour("#gggggg")))
  expect_false(unname(is_colour("#12345")))
  expect_false(unname(is_colour("#1234567")))
  expect_false(unname(is_colour("notacolor")))
  expect_false(unname(is_colour("")))

  # Mixed valid and invalid
  mixed_colors <- c("#ff00ff", "#gggggg", "red", "notacolor")
  expect_equal(unname(is_colour(mixed_colors)), c(TRUE, FALSE, TRUE, FALSE))
})

test_that("is_colour handles non-character input", {
  expect_equal(is_colour(123), FALSE)
  expect_equal(is_colour(NULL), FALSE)
  expect_equal(is_colour(TRUE), FALSE)
  expect_equal(is_colour(c(1, 2, 3)), FALSE)
})

test_that("hex_bw returns appropriate contrast colours", {
  # Light colours should return black text
  expect_equal(hex_bw("#ffffff"), "#000000") # White -> Black
  expect_equal(hex_bw("#ffff00"), "#000000") # Yellow -> Black

  # Dark colours should return white text
  expect_equal(hex_bw("#000000"), "#ffffff") # Black -> White
  expect_equal(hex_bw("#0000ff"), "#ffffff") # Blue -> White

  # Test with multiple colours
  colors <- c("#ffffff", "#000000", "#ff0000")
  result <- hex_bw(colors)
  expect_length(result, 3)
  expect_true(all(result %in% c("#000000", "#ffffff")))
})

test_that("hex_bw handles NA values", {
  # NA should return white
  expect_equal(hex_bw(NA_character_), "#ffffff")

  # Mixed with NA
  colors <- c("#ffffff", NA_character_, "#000000")
  result <- hex_bw(colors)
  expect_equal(result, c("#000000", "#ffffff", "#ffffff"))
})

test_that("check_colour_palette validates colour palettes", {
  # Valid colour palettes should not error
  expect_no_error(check_colour_palette(c("#ff0000", "#00ff00", "#0000ff")))
  expect_no_error(check_colour_palette(c("red", "green", "blue")))
  expect_no_error(check_colour_palette(NULL))

  # Functions are handled but may error in validation - test in get_remaining_colours
})

test_that("check_colour_palette rejects invalid palettes", {
  # Invalid colours should error
  expect_error(
    check_colour_palette(c("#invalid", "#ff0000")),
    "Invalid user-specified colours"
  )
  expect_error(
    check_colour_palette(c("notacolor", "red")),
    "Invalid user-specified colours"
  )
  expect_error(check_colour_palette(123), "Invalid user-specified colours")
})

test_that("get_remaining_colours handles user colour sets", {
  # Simple case: enough colours provided
  user_colors <- c("#ff0000", "#00ff00", "#0000ff", "#ffff00")
  result <- get_remaining_colours(user_colors, 3, ordinal = FALSE)
  expect_equal(result, c("#ff0000", "#00ff00", "#0000ff"))
  expect_length(result, 3)
})

test_that("get_remaining_colours handles colour functions", {
  # The function needs to be tested differently as check_colour_palette
  # doesn't properly handle functions - this reveals a limitation
  color_function <- function() c("#ff0000", "#00ff00", "#0000ff")

  # Test that function is called and processed correctly despite validation issue
  expect_error(
    get_remaining_colours(color_function, 2, ordinal = FALSE),
    "cannot coerce type 'closure'"
  )
})

test_that("get_remaining_colours handles insufficient colours", {
  # Not enough colours provided
  user_colors <- c("#ff0000", "#00ff00")
  expect_warning(
    result <- get_remaining_colours(user_colors, 5, ordinal = FALSE),
    "Fewer colours in user-provided colour palette than needed"
  )
})

test_that("get_remaining_colours handles ordinal option", {
  # This function uses the internal subset_vector function

  user_colors <- c("#ff0000", "#00ff00", "#0000ff", "#ffff00", "#ff00ff")

  # Test if ordinal = TRUE uses spread algorithm
  # This test might need adjustment based on subset_vector implementation
  expect_no_error(get_remaining_colours(user_colors, 3, ordinal = TRUE))
})

test_that("get_remaining_colours handles NULL input", {
  expect_no_error(result <- get_remaining_colours(NULL, 3, ordinal = FALSE))
})
