#' Convert a CDS position to the genomic position
#'
#' @param cds.position A positive integer specifying the position in the coding sequence of a transcript
#' @param tx.id The Ensmble ID of the transcript
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' 
#' @return A genomic position
#' 
#' @importFrom ensembldb cdsBy
#' @importFrom ensembldb cdsToTranscript
#' @importFrom ensembldb transcriptToGenome
#' @importFrom IRanges ranges
#' @importFrom IRanges IRanges
#' @importFrom BiocGenerics width
#' @importFrom BiocGenerics start
#' 
#' @examples
#' \dontrun{
#'     cdsToGenome()
#' }
#'
#' @noRd
cdsToGenome <- function(cds.position, tx.id, ensdb){
    
    cds.list <- ensembldb::cdsBy(ensdb, filter = ~ tx_id==tx.id)[[1]]
    cds.length <- sum(BiocGenerics::width(IRanges::ranges(cds.list)))
    
    if( cds.position > cds.length ){
        dna.position <- NA
    }else{
        cds.range <- IRanges::IRanges(start = cds.position, end = cds.position, names = tx.id)
        tx.range <- ensembldb::cdsToTranscript(cds.range, ensdb)
        dna.range <- ensembldb::transcriptToGenome(tx.range, ensdb)[[1]]
        dna.position <- BiocGenerics::start(IRanges::ranges(dna.range))
    }
    
    return(dna.position)
}

