% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rank_stratified.R
\name{rank_stratified}
\alias{rank_stratified}
\title{Stratified hierarchical ranking across multiple variables}
\usage{
rank_stratified(
  data,
  cols = NULL,
  sort_by = "frequency",
  desc = FALSE,
  ties.method = "average",
  na.last = TRUE,
  freq_tiebreak = "match_desc",
  verbose = TRUE
)
}
\arguments{
\item{data}{A data frame. Each selected column
represents one level of the stratified hierarchy, in the order given by
\code{cols}.}

\item{cols}{Optional column specification indicating which variables in \code{data}
to use for ranking, and in what order. Can be:
\itemize{
\item \code{NULL} (default): use all columns of \code{data} in their existing order.
\item A character vector of column names.
\item An integer vector of column positions.
}}

\item{sort_by}{Character scalar or vector specifying how to rank each
non-numeric column. Each element must be either \code{"alphabetical"} or
\code{"frequency"}, matching the behaviour of \code{\link[=smartrank]{smartrank()}}. If a single
value is supplied it is recycled for all columns. For numeric columns,
\code{sort_by} is ignored and ranking is always based on numeric order.}

\item{desc}{Logical scalar or vector indicating whether to rank each column
in descending order. If a single value is supplied it is recycled for all
columns.}

\item{ties.method}{Passed to \code{\link[base:rank]{base::rank()}} when resolving ties at each
level; must be one of \code{"average"}, \code{"first"}, \code{"last"}, \code{"random"},
\code{"max"}, or \code{"min"}. See \code{\link[base:rank]{base::rank()}} for details.}

\item{na.last}{Logical, controlling the treatment of missing values,
as in \code{\link[base:rank]{base::rank()}}. If \code{TRUE}, \code{NA}s are given the largest ranks; if
\code{FALSE}, the smallest. Unlike \code{\link[base:rank]{base::rank()}} or \code{\link[=smartrank]{smartrank()}}, \code{na.last}
cannot be set to \code{NA} in \code{rank_stratified()}, because dropping rows would
change group membership and break stratified ranking.}

\item{freq_tiebreak}{Character scalar or vector controlling how
alphabetical tie-breaking works when \code{sort_by = "frequency"} and the
column is character/factor/logical. Each element must be one of:
\itemize{
\item \code{"match_desc"} (default): alphabetical tie-breaking follows
\code{desc} for that column (ascending when \code{desc = FALSE}, descending
when \code{desc = TRUE}).
\item \code{"asc"}: ties are always broken by ascending alphabetical order.
\item \code{"desc"}: ties are always broken by descending alphabetical order.
}
If a single value is supplied, it is recycled for all columns.}

\item{verbose}{Logical; if \code{TRUE}, emit messages when \code{sort_by} is ignored
(e.g. for numeric columns), mirroring the behaviour of \code{\link[=smartrank]{smartrank()}}.}
}
\value{
A numeric vector of length \code{nrow(data)}, containing stratified ranks.
Smaller values indicate "earlier" rows in the stratified hierarchy.
}
\description{
\code{rank_stratified()} computes a single, combined rank for each row of a
data frame using \strong{stratified hierarchical ranking}.
The first variable is ranked globally; each subsequent variable is then
ranked \strong{within strata defined by all previous variables}.
}
\details{
This is useful when you want a "truly hierarchical" ordering where,
for example, rows are first grouped and ordered by the frequency of
\code{gender}, and then within each \code{gender} group, ordered by the frequency
of \code{pet} \strong{within that gender}, rather than globally.

The result is a single rank vector that can be passed directly to
\code{\link[base:order]{base::order()}} to obtain a stratified, multi-level
ordering.

Stratified ranking proceeds level by level:

\enumerate{
\item The first selected column is ranked globally, using \code{sort_by[1]}
(for non-numeric) and \code{desc[1]}.
\item For the second column, ranks are computed \strong{separately within each
distinct combination of values of all previous columns}. Within each
stratum, the second column is ranked using \code{sort_by[2]} / \code{desc[2]}.
\item This process continues for each subsequent column: at level \emph{k},
ranking is done within strata defined by columns 1, 2, ..., \emph{k-1}.
}

This yields a single composite rank per row that reflects a "true"
hierarchical (i.e. stratified) ordering: earlier variables define strata, and later variables
are only compared \strong{within} those strata (for example, by within-stratum
frequency).
}
\examples{
library(rank)

data <- data.frame(
  gender = c("male", "male", "male", "male", "female", "female", "male", "female"),
  pet    = c("cat",  "cat",  "magpie", "magpie", "giraffe", "cat", "giraffe", "cat")
)

# Stratified ranking: first by gender frequency, then within each gender
# by pet frequency *within that gender*
r <- rank_stratified(
  data,
  cols = c("gender", "pet"),
  sort_by = c("frequency", "frequency"),
  desc = TRUE
)

data[order(r), ]

}
