\name{fit.pweibull}
\alias{fit.pweibull}

\title{
Parameter Estimation for the Power Piecewise Weibull Model
}

\description{
Fits the power piecewise Weibull model and returns parameter estimates,
standard errors, likelihood measures, and the selected time partition.
}

\usage{
fit.pweibull(formula, data, L = 1, t = NULL, prec = 1e-04, max.iter = 1000,
    lambda.identical = FALSE, alpha.identical = FALSE, alpha.fixed = FALSE)
}

\arguments{
  \item{formula}{
A model formula of class \code{"formula"} describing the survival model
to be fitted. Details on model specification are given in \emph{Details}.
}

  \item{data}{
An optional data frame, list, or environment containing the variables in the
model. If not found in \code{data}, variables are taken from
\code{environment(formula)}.
}

  \item{L}{
Number of partitions to be used. Ignored if \code{t} is specified.
}

  \item{t}{
A fixed time partition. Ignored if \code{L} is supplied.
}

  \item{prec}{
Numerical tolerance used in the estimation procedure (default \code{1e-4}).
}

  \item{max.iter}{
Maximum number of iterations allowed in the estimation algorithm
(default \code{1000}).
}

  \item{lambda.identical}{
Logical; should the \eqn{\lambda}'s be constrained to be equal?
(default \code{FALSE}).
}

  \item{alpha.identical}{
Logical; should the \eqn{\alpha}'s be constrained to be equal?
(default \code{FALSE}).
}

  \item{alpha.fixed}{
Should \eqn{\alpha} be fixed at a known value?  
If \code{FALSE} (default), \eqn{\alpha} is estimated.  
If a positive number is provided, all \eqn{\alpha_\ell} are fixed at that value.
}
}

\details{
The hazard function of the power piecewise Weibull model is given by
\deqn{
h(t \mid \boldsymbol{\lambda}, \boldsymbol{\alpha})
  = \lambda_\ell \alpha_\ell t^{\alpha_\ell - 1},
  \qquad t \in (a_{\ell-1}, a_\ell),\; \ell = 1, \ldots, L,
}
where \eqn{0 = a_0 < a_1 < \cdots < a_L < \infty} defines the time partition,
\eqn{\boldsymbol{\lambda} = (\lambda_1, \ldots, \lambda_L)} and
\eqn{\boldsymbol{\alpha} = (\alpha_1, \ldots, \alpha_L)}.

Special cases include:

\itemize{
\item \eqn{\alpha_1 = \cdots = \alpha_L = 1}: the piecewise exponential model
(Feigl and Zelen, 1965; Friedman, 1982),
\item \eqn{\alpha_1 = \cdots = \alpha_L = 2}: a piecewise Rayleigh model.
}
}

\value{
A list with components:

\item{estimate}{A matrix containing parameter estimates and corresponding standard errors.}
\item{logLik}{Log-likelihood evaluated at the estimated parameters.}
\item{t}{The time partition used.}
\item{AIC}{Akaike Information Criterion.}
\item{BIC}{Bayesian Information Criterion.}
}

\references{
Feigl P., Zelen M. (1965).
Estimation of exponential survival probabilities with concomitant information.
\emph{Biometrics}, \bold{21}, 826-838.

Friedman M. (1982). 
Piecewise exponential models for survival data with covariates.
\emph{Annals of Statistics}, \bold{10}, 101-113.

Gomez Y. M., Gallardo D. I., Arnold B. C. (2018).
The power piecewise exponential model.
\emph{Journal of Statistical Computation and Simulation}, \bold{88}, 825-840.
}

\author{
Diego I. Gallardo, Yolanda M. Gomez, Hector W. Gomez, and Barry C. Arnold.
}

\examples{
\donttest{
library(survival)
set.seed(3100)

n  <- 200
x1 <- rnorm(n)
x2 <- rnorm(n)

## design matrix
x  <- model.matrix(~ x1 + x2)[, -1]

lambda <- c(0.05, 0.03)
rate   <- exp(cbind(x1, x2) \%*\% c(0.5, -0.5))

time <- numeric(n)
for (i in 1:n)
  time[i] <- rpweibull(1, rate = lambda * rate[i], alpha = c(1, 1), t = c(0, 10))

delta <- rep(1, n)

data <- data.frame(time = time, delta = delta, x1 = x1, x2 = x2)

fit.pweibull(survival::Surv(time, delta) ~ x1 + x2, data = data, L = 2)
}
}
