% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/llm_batch.R
\name{llm_submit_pairs_batch}
\alias{llm_submit_pairs_batch}
\title{Submit pairs to an LLM backend via batch API}
\usage{
llm_submit_pairs_batch(
  pairs,
  backend = c("openai", "anthropic", "gemini"),
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  include_thoughts = FALSE,
  include_raw = FALSE,
  ...
)
}
\arguments{
\item{pairs}{A data frame or tibble of pairs with columns \code{ID1}, \code{text1},
\code{ID2}, and \code{text2}. Additional columns are allowed and will be carried
through where supported.}

\item{backend}{Character scalar; one of \code{"openai"}, \code{"anthropic"}, or
\code{"gemini"}. Matching is case-insensitive.}

\item{model}{Character scalar model name to use for the batch job.
\itemize{
\item For \code{"openai"}, use models like \code{"gpt-4.1"}, \code{"gpt-5.1"}, or \code{"gpt-5.2"}
(including date-stamped versions like \code{"gpt-5.2-2025-12-11"}).
\item For \code{"anthropic"}, use provider names like \code{"claude-4-5-sonnet"}
or date-stamped versions like \code{"claude-sonnet-4-5-20250929"}.
\item For \code{"gemini"}, use names like \code{"gemini-3-pro-preview"}.
}}

\item{trait_name}{A short name for the trait being evaluated (e.g.
\code{"overall_quality"}).}

\item{trait_description}{A human-readable description of the trait.}

\item{prompt_template}{A prompt template created by \code{\link[=set_prompt_template]{set_prompt_template()}}
or a compatible character scalar.}

\item{include_thoughts}{Logical; whether to request and parse model
"thoughts" (where supported).
\itemize{
\item For OpenAI GPT-5.1/5.2, setting this to \code{TRUE} defaults to the
\code{responses} endpoint.
\item For Anthropic, setting this to \code{TRUE} implies \code{reasoning = "enabled"}
(unless overridden) and sets \code{temperature = 1}.
}}

\item{include_raw}{Logical; whether to include raw provider responses in the
result (where supported by backends).}

\item{...}{Additional arguments passed through to the backend-specific
\verb{run_*_batch_pipeline()} functions. This can include provider-specific
options such as temperature or batch configuration fields. For OpenAI,
this may include \code{endpoint}, \code{temperature}, \code{top_p}, \code{logprobs},
\code{reasoning}, etc. For Anthropic, this may include \code{reasoning},
\code{max_tokens}, \code{temperature}, or \code{thinking_budget_tokens}.}
}
\value{
A list of class \code{"pairwiseLLM_batch"} containing at least:
\itemize{
\item \code{backend}: the backend identifier (\code{"openai"}, \code{"anthropic"}, \code{"gemini"}),
\item \code{batch_input_path}: path to the JSONL request file (if applicable),
\item \code{batch_output_path}: path to the JSONL output file (if applicable),
\item \code{batch}: provider-specific batch object (e.g., job metadata),
\item \code{results}: a tibble of parsed comparison results in the standard
pairwiseLLM schema.
}

Additional fields returned by the backend-specific pipeline functions are
preserved.
}
\description{
\code{llm_submit_pairs_batch()} is a backend-agnostic front-end for running
provider batch pipelines (OpenAI, Anthropic, Gemini). Together.ai and Ollama
are supported only for live comparisons.

It mirrors \code{\link[=submit_llm_pairs]{submit_llm_pairs()}} but uses the provider batch APIs under the
hood via \code{run_openai_batch_pipeline()}, \code{run_anthropic_batch_pipeline()},
and \code{run_gemini_batch_pipeline()}.

For OpenAI, this helper will by default:
\itemize{
\item Use the \code{chat.completions} batch style for most models, and
\item Automatically switch to the \code{responses} style endpoint when:
\itemize{
\item \code{model} starts with \code{"gpt-5.1"} or \code{"gpt-5.2"} (including date-stamped
versions like \code{"gpt-5.2-2025-12-11"}) and
\item either \code{include_thoughts = TRUE} \strong{or} a non-\code{"none"} \code{reasoning}
effort is supplied in \code{...}.
}
}

\strong{Temperature Defaults:}
For OpenAI, if \code{temperature} is not specified in \code{...}:
\itemize{
\item It defaults to \code{0} (deterministic) for standard models or when reasoning is
disabled (\code{reasoning = "none"}) on supported models (5.1/5.2).
\item It remains \code{NULL} (API default) when reasoning is enabled, as the API
does not support temperature with reasoning.
}

For Anthropic, standard and date-stamped model names
(e.g. \code{"claude-sonnet-4-5-20250929"}) are supported. This helper delegates
temperature and extended-thinking behaviour to
\code{\link[=run_anthropic_batch_pipeline]{run_anthropic_batch_pipeline()}} and \code{\link[=build_anthropic_batch_requests]{build_anthropic_batch_requests()}},
which apply the following rules:
\itemize{
\item When \code{reasoning = "none"} (no extended thinking), the default
temperature is \code{0} (deterministic) unless you explicitly supply a
different \code{temperature} in \code{...}.
\item When \code{reasoning = "enabled"} (extended thinking), Anthropic requires
\code{temperature = 1}. If you supply a different value in \code{...}, an error
is raised. Default values in this mode are \code{max_tokens = 2048} and
\code{thinking_budget_tokens = 1024}, subject to
\verb{1024 <= thinking_budget_tokens < max_tokens}.
\item Setting \code{include_thoughts = TRUE} while leaving \code{reasoning = "none"}
causes \code{run_anthropic_batch_pipeline()} to upgrade to
\code{reasoning = "enabled"}, which implies \code{temperature = 1} for the batch.
}

For Gemini, this helper simply forwards \code{include_thoughts} and other
arguments to \code{\link[=run_gemini_batch_pipeline]{run_gemini_batch_pipeline()}}, which is responsible for
interpreting any thinking-related options.

Currently, this function \emph{synchronously} runs the full batch pipeline for
each backend (build requests, create batch, poll until complete, download
results, parse). The returned object contains both metadata and a normalized
\code{results} tibble. See \code{\link[=llm_download_batch_results]{llm_download_batch_results()}} to extract the results.
}
\examples{
# Requires:
# - Internet access
# - Provider API key set in your environment (OPENAI_API_KEY /
#   ANTHROPIC_API_KEY / GEMINI_API_KEY)
# - Billable API usage
\dontrun{
pairs <- tibble::tibble(
  ID1   = c("S01", "S03"),
  text1 = c("Text 1", "Text 3"),
  ID2   = c("S02", "S04"),
  text2 = c("Text 2", "Text 4")
)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# OpenAI batch
batch_openai <- llm_submit_pairs_batch(
  pairs             = pairs,
  backend           = "openai",
  model             = "gpt-4.1",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  include_thoughts  = FALSE
)
res_openai <- llm_download_batch_results(batch_openai)

# Anthropic batch
batch_anthropic <- llm_submit_pairs_batch(
  pairs             = pairs,
  backend           = "anthropic",
  model             = "claude-4-5-sonnet",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  include_thoughts  = FALSE
)
res_anthropic <- llm_download_batch_results(batch_anthropic)

# Gemini batch
batch_gemini <- llm_submit_pairs_batch(
  pairs             = pairs,
  backend           = "gemini",
  model             = "gemini-3-pro-preview",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  include_thoughts  = TRUE
)
res_gemini <- llm_download_batch_results(batch_gemini)
}

}
