% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mr_mash_rss.R
\name{mr.mash.rss}
\alias{mr.mash.rss}
\title{Multiple Regression with Multivariate Adaptive Shrinkage
 from summary data.}
\usage{
mr.mash.rss(
  Bhat,
  Shat,
  Z,
  R,
  covY,
  n,
  S0,
  w0 = rep(1/(length(S0)), length(S0)),
  V,
  mu1_init = NULL,
  tol = 0.0001,
  convergence_criterion = c("mu1", "ELBO"),
  max_iter = 5000,
  update_w0 = TRUE,
  update_w0_method = "EM",
  w0_penalty = rep(1, length(S0)),
  update_w0_max_iter = Inf,
  w0_threshold = 0,
  compute_ELBO = TRUE,
  standardize = FALSE,
  verbose = TRUE,
  update_V = FALSE,
  update_V_method = c("full", "diagonal"),
  version = c("Rcpp", "R"),
  e = 1e-08,
  ca_update_order = c("consecutive", "decreasing_logBF", "increasing_logBF", "random"),
  X_colmeans = NULL,
  Y_colmeans = NULL,
  check_R = TRUE,
  R_tol = 1e-08,
  nthreads = as.integer(NA)
)
}
\arguments{
\item{Bhat}{p x r matrix of regression coefficients from univariate
simple linear regression.}

\item{Shat}{p x r matrix of standard errors of the regression coefficients 
from univariate simple linear regression.}

\item{Z}{p x r matrix of Z-scores from univariate
simple linear regression.}

\item{R}{p x p dense or sparse correlation matrix among the variables.}

\item{covY}{r x r covariance matrix across responses.}

\item{n}{scalar indicating the sample size.}

\item{S0}{List of length K containing the desired r x r prior
covariance matrices on the regression coefficients.}

\item{w0}{K-vector with prior mixture weights, each associated with
the respective covariance matrix in \code{S0}.}

\item{V}{r x r residual covariance matrix.}

\item{mu1_init}{p x r matrix of initial estimates of the posterior
mean regression coefficients. These should be on the same scale as
the X provided. If \code{standardize=TRUE}, mu1_init will be scaled
appropriately after standardizing X.}

\item{tol}{Convergence tolerance.}

\item{convergence_criterion}{Criterion to use for convergence check.}

\item{max_iter}{Maximum number of iterations for the optimization
algorithm.}

\item{update_w0}{If \code{TRUE}, prior weights are updated.}

\item{update_w0_method}{Method to update prior weights. Only EM is
currently supported.}

\item{w0_penalty}{K-vector of penalty terms (>=1) for each 
mixture component. Default is all components are unpenalized.}

\item{update_w0_max_iter}{Maximum number of iterations for the update
of w0.}

\item{w0_threshold}{Drop mixture components with weight less than this value.
Components are dropped at each iteration after 15 initial iterations.
This is done to prevent from dropping some potentially important 
components prematurely.}

\item{compute_ELBO}{If \code{TRUE}, ELBO is computed.}

\item{standardize}{If \code{TRUE}, X is "standardized" using the
sample means and sample standard deviations. Standardizing X
allows a faster implementation, but the prior has a different
interpretation. Coefficients and covariances are returned on the
original scale.}

\item{verbose}{If \code{TRUE}, some information about the
algorithm's process is printed at each iteration.}

\item{update_V}{if \code{TRUE}, residual covariance is updated.}

\item{update_V_method}{Method to update residual covariance. So far,
"full" and "diagonal" are supported. If \code{update_V=TRUE} and V 
is not provided by the user, this option will determine how V is 
computed (and fixed) internally from \code{mu1_init}.}

\item{version}{Whether to use R or C++ code to perform the
coordinate ascent updates.}

\item{e}{A small number to add to the diagonal elements of the
prior matrices to improve numerical stability of the updates.}

\item{ca_update_order}{The order with which coordinates are
updated.  So far, "consecutive", "decreasing_logBF",
"increasing_logBF", "random" are supported.}

\item{X_colmeans}{a p-vector of variable means.}

\item{Y_colmeans}{a r-vector of response means.}

\item{check_R}{If \code{TRUE}, R is checked to be positive semidefinite.}

\item{R_tol}{tolerance to declare positive semi-definiteness of R.}

\item{nthreads}{Number of RcppParallel threads to use for the
updates. When \code{nthreads} is \code{NA}, the default number of
threads is used; see
\code{\link[RcppParallel]{defaultNumThreads}}. This setting is
ignored when \code{version = "R"}.}
}
\value{
A mr.mash.rss fit, stored as a list with some or all of the
following elements:

\item{mu1}{p x r matrix of posterior means for the regression
  coeffcients.}

\item{S1}{r x r x p array of posterior covariances for the
  regression coeffcients.}

\item{w1}{p x K matrix of posterior assignment probabilities to the
  mixture components.}
  
\item{V}{r x r residual covariance matrix}

\item{w0}{K-vector with (updated, if \code{update_w0=TRUE}) prior mixture weights, each associated with
  the respective covariance matrix in \code{S0}}.
  
\item{S0}{r x r x K array of prior covariance matrices
  on the regression coefficients}.

\item{intercept}{r-vector containing posterior mean estimate of the
  intercept, if \code{X_colmeans} and \code{Y_colmeans} are provided.
  Otherwise, \code{NA} is output.}

\item{ELBO}{Evidence Lower Bound (ELBO) at last iteration.}

\item{progress}{A data frame including information regarding
  convergence criteria at each iteration.}
  
\item{converged}{\code{TRUE} or \code{FALSE}, indicating whether
  the optimization algorithm converged to a solution within the chosen tolerance
  level.}
  
\item{Y}{n x r matrix of responses at last iteration (only relevant when missing values
  are present in the input Y).}
}
\description{
Performs multivariate multiple regression with
  mixture-of-normals prior.
}
\examples{
###Set seed
set.seed(123)

###Simulate X and Y
##Set parameters
n  <- 1000
p <- 100
p_causal <- 20
r <- 5

###Simulate data
out <- simulate_mr_mash_data(n, p, p_causal, r, pve=0.5, B_cor=1,
                             B_scale=1, X_cor=0, X_scale=1, V_cor=0)

###Split the data in training and test sets
Ytrain <- out$Y[-c(1:200), ]
Xtrain <- out$X[-c(1:200), ]
Ytest <- out$Y[c(1:200), ]
Xtest <- out$X[c(1:200), ]

###Specify the covariance matrices for the mixture-of-normals prior.
univ_sumstats <- compute_univariate_sumstats(Xtrain, Ytrain,
                   standardize=TRUE, standardize.response=FALSE)
grid <- autoselect.mixsd(univ_sumstats, mult=sqrt(2))^2
S0 <- compute_canonical_covs(ncol(Ytrain), singletons=TRUE,
                             hetgrid=c(0, 0.25, 0.5, 0.75, 1))
S0 <- expand_covs(S0, grid, zeromat=TRUE)

###Fit mr.mash
# Note that max_iter was set to 4 in this example to shorten
# the running time. In practice, you should allow the model-fitting
# algorithm to run for many more iterations to obtain better estimates.
covY <- cov(Ytrain)
corX <- cor(Xtrain)
n_train <- nrow(Ytrain)
fit <- mr.mash.rss(Bhat=univ_sumstats$Bhat, Shat=univ_sumstats$Shat,
                   S0=S0, covY=covY, R=corX, n=n_train, V=covY,
                   update_V=TRUE, X_colmeans=colMeans(Xtrain),
                   Y_colmeans=colMeans(Ytrain), max_iter = 4,
                   nthreads = 1)

# Predict the multivariate outcomes in the test set using the fitted model.
Ytest_est <- predict(fit,Xtest)
plot(Ytest_est,Ytest,pch = 20,col = "darkblue",xlab = "true",
     ylab = "predicted")
abline(a = 0,b = 1,col = "magenta",lty = "dotted")

}
