#include <cstdint>
#include <cstring>
#include <sys/stat.h>
#include <sys/types.h>

#include "GenomeTrackSparse.h"
#include "TrackIndex.h"

const int GenomeTrackSparse::RECORD_SIZE = 2 * sizeof(int64_t) + sizeof(float);
constexpr size_t GenomeTrackSparse::kSparseRecBytes;

GenomeTrackSparse::GenomeTrackSparse() :
	GenomeTrack1D(SPARSE),
	m_loaded(false),
	m_num_records(0),
	m_last_min_pos(numeric_limits<double>::quiet_NaN())
{}

void GenomeTrackSparse::sync_master_state_from_dependent()
{
	if (!m_master_obj)
		return;

	for (size_t i = 0; i < m_functions.size(); ++i) {
		if (m_functions[i] && !m_master_obj->m_functions[i]) {
			m_master_obj->m_functions[i] = true;
			m_master_obj->m_fast_path_mode = 0;
		}
	}

	if (m_use_quantile && !m_master_obj->m_use_quantile) {
		m_master_obj->m_use_quantile = true;
		m_master_obj->m_sp = m_sp;
		m_master_obj->m_fast_path_mode = 0;
	}
}

void GenomeTrackSparse::copy_state_from_master()
{
	if (!m_master_obj)
		return;

	if (m_functions[AVG])
		m_last_avg = m_master_obj->m_last_avg;
	if (m_functions[MIN])
		m_last_min = m_master_obj->m_last_min;
	if (m_functions[MAX])
		m_last_max = m_master_obj->m_last_max;
	if (m_functions[MAX_POS])
		m_last_max_pos = m_master_obj->m_last_max_pos;
	if (m_functions[MIN_POS])
		m_last_min_pos = m_master_obj->m_last_min_pos;
	if (m_functions[NEAREST])
		m_last_nearest = m_master_obj->m_last_nearest;
	if (m_functions[STDDEV])
		m_last_stddev = m_master_obj->m_last_stddev;
	if (m_functions[SUM])
		m_last_sum = m_master_obj->m_last_sum;
	if (m_functions[LSE])
		m_last_lse = m_master_obj->m_last_lse;
	if (m_functions[EXISTS])
		m_last_exists = m_master_obj->m_last_exists;
	if (m_functions[SIZE])
		m_last_size = m_master_obj->m_last_size;
	if (m_functions[SAMPLE])
		m_last_sample = m_master_obj->m_last_sample;
	if (m_functions[SAMPLE_POS])
		m_last_sample_pos = m_master_obj->m_last_sample_pos;
	if (m_functions[FIRST])
		m_last_first = m_master_obj->m_last_first;
	if (m_functions[FIRST_POS])
		m_last_first_pos = m_master_obj->m_last_first_pos;
	if (m_functions[LAST])
		m_last_last = m_master_obj->m_last_last;
	if (m_functions[LAST_POS])
		m_last_last_pos = m_master_obj->m_last_last_pos;
	if (m_use_quantile)
		m_sp = m_master_obj->m_sp;
}

void GenomeTrackSparse::read_header_at_current_pos_(BufferedFile &bf)
{
	int32_t format_signature = 0;
	if (bf.read(&format_signature, sizeof(format_signature)) != sizeof(format_signature))
		TGLError<GenomeTrackSparse>("Corrupt sparse header in %s", bf.file_name().c_str());
	if (format_signature >= 0)
		TGLError<GenomeTrackSparse>("Invalid sparse header signature in %s", bf.file_name().c_str());
}

void GenomeTrackSparse::init_read(const char *filename, int chromid)
{
	m_loaded = false; // Critical for read_file_into_mem()
	m_fast_path_mode = 0;
	uint64_t header_start = 0;
	uint64_t total_bytes = 0;

	// Check for indexed format FIRST
	const std::string track_dir = GenomeTrack::get_track_dir(filename);
	const std::string idx_path = track_dir + "/track.idx";

	struct stat idx_st;
	if (stat(idx_path.c_str(), &idx_st) == 0) {
		// --- INDEXED PATH ---
		const std::string dat_path  = track_dir + "/track.dat";

		// "Smart Handle" logic - reopen if path or mode changed
		if (!m_dat_open || m_dat_path != dat_path || m_dat_mode != "rb") {
			m_bfile.close();
			if (m_bfile.open(dat_path.c_str(), "rb"))
				TGLError<GenomeTrackSparse>("Cannot open %s: %s", dat_path.c_str(), strerror(errno));
			m_dat_open = true;
			m_dat_path = dat_path;
			m_dat_mode = "rb";
		}

		auto idx   = get_track_index(track_dir);
		if (!idx)
			TGLError<GenomeTrackSparse>("Failed to load track index for %s", track_dir.c_str());

		auto entry = idx->get_entry(chromid);
		if (!entry || entry->length == 0) {
			// Chromosome not in index or empty contig - treat as empty
			m_num_records = 0;
			m_chromid = chromid;
			return;
		}

		if (m_bfile.seek(entry->offset, SEEK_SET))
			TGLError<GenomeTrackSparse>("Failed to seek to offset %llu in %s",
				(unsigned long long)entry->offset, dat_path.c_str());

		header_start = entry->offset;
		read_header_at_current_pos_(m_bfile);
		total_bytes = entry->length;
	} else {
		// --- PER-CHROMOSOME PATH ---
		m_bfile.close(); // Close previous file (if any)
		m_dat_open = false;
		read_type(filename); // This opens m_bfile and reads header

		header_start = 0;
		total_bytes = m_bfile.file_size();
	}

	// --- COMMON LOGIC ---
	const uint64_t header_size  = m_bfile.tell() - header_start;
	const double n = (total_bytes - header_size) / (double)kSparseRecBytes;

	if (n != (int64_t)n)
		TGLError<GenomeTrackSparse>("Invalid sparse track file %s: file size (%llu bytes) minus header (%llu bytes) is not divisible by record size (%zu bytes). Computed %.2f records.",
			filename, (unsigned long long)total_bytes, (unsigned long long)header_size, kSparseRecBytes, n);

	m_num_records = (int64_t)n;
	m_chromid = chromid;
}

void GenomeTrackSparse::init_write(const char *filename, int chromid)
{
	m_bfile.close();
	m_loaded = false;
	m_fast_path_mode = 0;
	write_type(filename);
	m_chromid = chromid;
}

void GenomeTrackSparse::read_file_into_mem()
{
	if (m_loaded)
		return;

	m_intervals.resize(m_num_records);
	m_vals.resize(m_num_records);

	if (m_num_records == 0) {
		m_icur_interval = m_intervals.begin();
		m_loaded = true;
		return;
	}

	static const int64_t CHUNK_RECORDS = 32768;
	const size_t chunk_capacity_bytes = (size_t)CHUNK_RECORDS * kSparseRecBytes;
	if (m_read_chunk.size() < chunk_capacity_bytes)
		m_read_chunk.resize(chunk_capacity_bytes);

	int64_t out_idx = 0;
	while (out_idx < m_num_records) {
		int64_t chunk_records = std::min<int64_t>(CHUNK_RECORDS, m_num_records - out_idx);
		size_t bytes_to_read = (size_t)chunk_records * kSparseRecBytes;

		uint64_t bytes_read = m_bfile.read(m_read_chunk.data(), bytes_to_read);
		if (bytes_read != bytes_to_read) {
			if (m_bfile.error())
				TGLError<GenomeTrackSparse>("Failed to read sparse track file %s: %s", m_bfile.file_name().c_str(), strerror(errno));
			TGLError<GenomeTrackSparse>("Truncated sparse track file %s: expected %zu bytes in chunk, but only read %llu bytes",
				m_bfile.file_name().c_str(), bytes_to_read, (unsigned long long)bytes_read);
		}

		const char *ptr = m_read_chunk.data();
		for (int64_t i = 0; i < chunk_records; ++i, ++out_idx) {
			GInterval &interval = m_intervals[out_idx];
			memcpy(&interval.start, ptr, sizeof(int64_t));
			ptr += sizeof(int64_t);
			memcpy(&interval.end, ptr, sizeof(int64_t));
			ptr += sizeof(int64_t);
			memcpy(&m_vals[out_idx], ptr, sizeof(float));
			ptr += sizeof(float);

			if (isinf(m_vals[out_idx]))
				m_vals[out_idx] = numeric_limits<float>::quiet_NaN();

			interval.chromid = m_chromid;

			if (interval.start < 0) {
				TGLError<GenomeTrackSparse>("Invalid sparse track file %s: interval %lld has negative start (%lld)",
					m_bfile.file_name().c_str(), (long long)out_idx, (long long)interval.start);
			}
			if (interval.start >= interval.end) {
				TGLError<GenomeTrackSparse>("Invalid sparse track file %s: interval %lld has start (%lld) >= end (%lld)",
					m_bfile.file_name().c_str(), (long long)out_idx, (long long)interval.start, (long long)interval.end);
			}
			if (out_idx && interval.start < m_intervals[out_idx - 1].end) {
				TGLError<GenomeTrackSparse>("Invalid sparse track file %s: interval %lld [%lld, %lld) overlaps with previous interval [%lld, %lld)",
					m_bfile.file_name().c_str(), (long long)out_idx,
					(long long)interval.start, (long long)interval.end,
					(long long)m_intervals[out_idx - 1].start, (long long)m_intervals[out_idx - 1].end);
			}
		}
	}

	m_icur_interval = m_intervals.begin();
	m_loaded = true;
}

void GenomeTrackSparse::read_interval(const GInterval &interval)
{
	if (m_master_obj) {
		if (!m_master_synced) {
			sync_master_state_from_dependent();
			m_master_synced = true;
		}
		m_master_obj->read_interval(interval);
		copy_state_from_master();
		return;
	}

	m_last_avg = m_last_nearest = m_last_min = m_last_max = m_last_stddev = m_last_sum = numeric_limits<float>::quiet_NaN();
	if (m_functions[LSE])
		m_last_lse = numeric_limits<float>::quiet_NaN();
	if (m_functions[MAX_POS])
		m_last_max_pos = numeric_limits<double>::quiet_NaN();
	if (m_functions[MIN_POS])
		m_last_min_pos = numeric_limits<double>::quiet_NaN();
	if (m_functions[EXISTS])
		m_last_exists = 0;
	if (m_functions[SIZE])
		m_last_size = 0;
	if (m_functions[SAMPLE])
		m_last_sample = numeric_limits<float>::quiet_NaN();
	if (m_functions[SAMPLE_POS])
		m_last_sample_pos = numeric_limits<double>::quiet_NaN();
	if (m_functions[FIRST])
		m_last_first = numeric_limits<float>::quiet_NaN();
	if (m_functions[FIRST_POS])
		m_last_first_pos = numeric_limits<double>::quiet_NaN();
	if (m_functions[LAST])
		m_last_last = numeric_limits<float>::quiet_NaN();
	if (m_functions[LAST_POS])
		m_last_last_pos = numeric_limits<double>::quiet_NaN();

	if (m_use_quantile)
		m_sp.reset();

	read_file_into_mem();

	if (m_intervals.empty())
		return;

	if (m_intervals.front().start >= interval.end) {
		m_last_nearest = m_vals.front();
		return;
	}

	if (m_intervals.back().end <= interval.start) {
		m_last_nearest = m_vals.back();
		return;
	}

	const bool avg_nearest_only = uses_avg_nearest_fast_path();

	if (check_first_overlap(m_icur_interval, interval)) {
		if (avg_nearest_only)
			calc_vals_avg_nearest_only(interval);
		else
			calc_vals(interval);
	} else if (m_icur_interval + 1 < m_intervals.end() && check_first_overlap(m_icur_interval + 1, interval)) {
		++m_icur_interval;
		if (avg_nearest_only)
			calc_vals_avg_nearest_only(interval);
		else
			calc_vals(interval);
	} else {
		// run the binary search
		GIntervals::const_iterator istart_interval = m_intervals.begin();
		GIntervals::const_iterator iend_interval = m_intervals.end();

		while (iend_interval - istart_interval > 1) {
			GIntervals::const_iterator imid_interval = istart_interval + (iend_interval - istart_interval) / 2;

			if (check_first_overlap(imid_interval, interval)) {
				m_icur_interval = imid_interval;
				if (avg_nearest_only)
					calc_vals_avg_nearest_only(interval);
				else
					calc_vals(interval);
				break;
			}

			// is mid_interval < interval?
			if (GIntervals::compare_by_start_coord(*imid_interval, interval))
				istart_interval = imid_interval;
			else
				iend_interval = imid_interval;
		}

		if (iend_interval - istart_interval == 1 && check_first_overlap(istart_interval, interval)) {
			m_icur_interval = istart_interval;
			if (avg_nearest_only)
				calc_vals_avg_nearest_only(interval);
			else
				calc_vals(interval);
		}

		if (iend_interval - istart_interval == 1)
			m_last_nearest = iend_interval == m_intervals.end() || interval.dist2interv(*istart_interval) <= interval.dist2interv(*iend_interval) ?
					m_vals[istart_interval - m_intervals.begin()] : m_vals[iend_interval - m_intervals.begin()];
	}
}

double GenomeTrackSparse::last_max_pos() const
{
	return m_last_max_pos;
}

double GenomeTrackSparse::last_min_pos() const
{
	return m_last_min_pos;
}

void GenomeTrackSparse::write_next_interval(const GInterval &interval, float val)
{
	uint64_t size = 0;
	size += m_bfile.write(&interval.start, sizeof(interval.start));
	size += m_bfile.write(&interval.end, sizeof(interval.end));
	size += m_bfile.write(&val, sizeof(val));

	if ((int)size != RECORD_SIZE) {
		if (m_bfile.error())
			TGLError<GenomeTrackSparse>("Failed to write a sparse track file %s: %s", m_bfile.file_name().c_str(), strerror(errno));
		TGLError<GenomeTrackSparse>("Failed to write a sparse track file %s", m_bfile.file_name().c_str());
	}
}

const GIntervals &GenomeTrackSparse::get_intervals()
{
	if (m_master_obj)
		return m_master_obj->get_intervals();
	read_file_into_mem();
	return m_intervals;
}

const vector<float> &GenomeTrackSparse::get_vals()
{
	if (m_master_obj)
		return m_master_obj->get_vals();
	read_file_into_mem();
	return m_vals;
}
