#' Generic summary function for bcmixmeta object in jarbes.
#'
#' @param object The object generated by the bcmixmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param show.clustering A logical value indicating whether to show the clustering parameters table. Defaults to `TRUE`.
#' @param show.neff A logical value indicating whether to show the effective sample size column (n.eff) and Rhat. Defaults to `TRUE`.
#'
#' @param ... \dots
#'
#' @export
summary.bcmixmeta = function(object, digits = 3, show.clustering = FALSE, show.neff = FALSE, ...) {
  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications from the fitted object
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Priors for the Model of Interest (Bias-Corrected)
  model.spec$hyper_corrected = list(
    mean.mu.theta = object$prior$mean.mu,
    sd.mu.theta = object$prior$sd.mu,
    scale.sigma.between = object$prior$scale.sigma.between,
    df.scale.between = object$prior$df.scale.between
  )

  # Priors for the Model of Bias (DPM part)
  model.spec$hyper_bias = list(
    scale.sigma.beta = object$prior$scale.sigma.beta,
    df.scale.beta = object$prior$df.scale.beta,
    B.lower = object$prior$B.lower,
    B.upper = object$prior$B.upper,
    a.0 = object$prior$a.0,
    a.1 = object$prior$a.1,
    alpha.0 = object$prior$alpha.0,
    alpha.1 = object$prior$alpha.1,
    K = object$prior$K
  )

  summary.m$model.specification = model.spec

  # Filter columns based on show.neff before extracting subsets
  if (!show.neff) {
    bugs.summary = bugs.summary[, !(colnames(bugs.summary) %in% c("n.eff", "Rhat")), drop = FALSE]
  }

  # Extract posterior distributions for main model parameters
  # The predictive effect is theta[N] where N is length(TE) + 1
  N_augmented = length(object$data$TE) + 1
  theta_pred_name = paste0("theta[", N_augmented, "]")
  param_names = c("mu.theta", theta_pred_name, "sd.theta") # sd.theta is tau in bcbnp_0
  param_names = intersect(param_names, rownames(bugs.summary))
  summary.m$model_corrected = bugs.summary[param_names, , drop = FALSE]

  # Rename rows for clarity
  row.names(summary.m$model_corrected) = c("Mean", "Predictive Effect", "SD (between studies sd)")

  # Extract bias model parameters
  bias_param_regex = "pi.B|alpha|K.hat|mu.beta|sd.beta"
  summary.m$model_biased = bugs.summary[grep(bias_param_regex, rownames(bugs.summary)), , drop = FALSE]

  # Rename rows for clarity
  rownames(summary.m$model_biased)[rownames(summary.m$model_biased) == "mu.beta"] = "Base Mean (mu.beta)"
  rownames(summary.m$model_biased)[rownames(summary.m$model_biased) == "sd.beta"] = "Base SD (sd.beta)"
  rownames(summary.m$model_biased)[rownames(summary.m$model_biased) == "pi.B"] = "Prob. of Bias (pi.B)"
  rownames(summary.m$model_biased)[rownames(summary.m$model_biased) == "alpha"] = "DP Conc. (alpha)"
  rownames(summary.m$model_biased)[rownames(summary.m$model_biased) == "K.hat"] = "No. Used Clusters (K.hat)"


  if (show.clustering) {
    # Extract posterior means for bias cluster parameters (mu.k and pi)
    mu_k_means = bugs.summary[grep("mu.k\\[.+\\]", rownames(bugs.summary)), "mean"]
    pi_k_means = bugs.summary[grep("pi\\[.+\\]", rownames(bugs.summary)), "mean"]

    # Ensure pi_k matches the number of mu_k
    if (length(mu_k_means) == length(pi_k_means)) {
      clustering_df = data.frame(
        mu.k = mu_k_means,
        pi.k = pi_k_means
      )
      num_clusters = length(mu_k_means)
      row.names(clustering_df) = paste0("Bias Cluster ", 1:num_clusters)
      summary.m$clustering_params = clustering_df
    }
  }

  # DIC and MCMC setup information
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD
  mcmc.setup = list(
    n.chains = bugs.output$n.chains,
    n.iter = bugs.output$n.iter,
    n.burnin = bugs.output$n.burnin,
    n.thin = object$BUGSoutput$n.thin # Fetch thin rate from BUGS output
  )
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.bcmixmeta"

  print(summary.m, digits, ...)
}

# ----------------------------------------------------------------------

#' @export
print.summary.bcmixmeta = function(x, digits = 3, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')

  cat('  Hyper-priors parameters for the Model of Interest (Bias-Corrected):\n')
  hyper_corrected = model.spec$hyper_corrected
  cat(paste('  Prior for mu.theta: Normal', '[', round(hyper_corrected$mean.mu.theta, digits), ', ', round(hyper_corrected$sd.mu.theta^2, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/sd.theta^2: Scale.Gamma', '[', round(hyper_corrected$scale.sigma.between, digits), ', ', round(hyper_corrected$df.scale.between, digits), ']', sep = ''))
  cat('\n')

  cat('\n')
  cat('  Hyper-priors parameters for the Model of Bias (DPM):\n')
  hyper_bias = model.spec$hyper_bias
  # mu.beta prior (mean of the DP base distribution)
  cat(paste('  Prior for mu.beta: Uniform', '[', round(hyper_bias$B.lower, digits), ', ', round(hyper_bias$B.upper, digits), ']', sep = ''))
  cat('\n')
  # 1/sd.beta^2 prior (precision of the DP base distribution)
  cat(paste('  Prior for 1/sd.beta^2: Scale.Gamma', '[', round(hyper_bias$scale.sigma.beta, digits), ', ', round(hyper_bias$df.scale.beta, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for pi.B: Beta', '[', round(hyper_bias$a.0, digits), ', ', round(hyper_bias$a.1, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for alpha: Uniform', '[', round(hyper_bias$alpha.0, digits), ', ', round(hyper_bias$alpha.1, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Maximum number of clusters (K): ', hyper_bias$K, sep = ''))
  cat('\n')

  cat('\n')
  cat('Posterior distributions:\n')
  cat('-------------------\n')
  cat('Model of Interest (Bias-Corrected):\n')
  print(round(x$model_corrected, digits))

  cat('\n-------------------\n')
  cat('Model of Bias:\n')
  print(round(x$model_biased, digits))

  if ("clustering_params" %in% names(x)) {
    cat('\n-------------------\n')
    cat('DPM Cluster Parameters (posterior means):\n')
    print.data.frame(round(x$clustering_params, digits), row.names = TRUE)
  }

  cat('\n-------------------\n')

  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' total iterations (first ', mcmc$n.burnin, ' discarded, thinning rate ', mcmc$n.thin, ')' , sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}
