% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtstm.R
\name{CohortDtstm}
\alias{CohortDtstm}
\title{Cohort discrete time state transition model}
\format{
An \link[R6:R6Class]{R6::R6Class} object.
}
\description{
Simulate outcomes from a cohort discrete time state transition model.
}
\examples{
library("data.table")
library("ggplot2")
theme_set(theme_bw())
set.seed(102)

# NOTE: This example replicates the "Simple Markov cohort model" 
# vignette using a different approach. Here, we explicitly construct
# the transition probabilities "by hand". In the vignette, the transition 
# probabilities are defined using expressions (i.e., by using 
# `define_model()`). The `define_model()` approach does (more or less) what 
# is done here under the hood.

# (0) Model setup
hesim_dat <- hesim_data(
  strategies = data.table(
    strategy_id = 1:2,
    strategy_name = c("Monotherapy", "Combination therapy")
  ),
  patients <- data.table(patient_id = 1),
  states = data.table(
    state_id = 1:3,
    state_name = c("State A", "State B", "State C")
  )
)
n_states <- nrow(hesim_dat$states) + 1
labs <- get_labels(hesim_dat)

# (1) Parameters
n_samples <- 10 # Number of samples for PSA

## Transition matrix
### Input data (one transition matrix for each parameter sample,
###             treatment strategy, patient, and time interval)
p_id <- tpmatrix_id(expand(hesim_dat, times = c(0, 2)), n_samples)
N <- nrow(p_id)

### Transition matrices (one for each row in p_id)
p <- array(NA, dim = c(n_states, n_states, nrow(p_id)))

#### Baseline risk
trans_mono <- rbind(
  c(1251, 350, 116, 17),
  c(0, 731, 512, 15),
  c(0, 0, 1312, 437),
  c(0, 0, 0, 469)
)
mono_ind <- which(p_id$strategy_id == 1 | p_id$time_id == 2)
p[,, mono_ind] <- rdirichlet_mat(n = 2, trans_mono)

#### Apply relative risks
combo_ind <- setdiff(1:nrow(p_id), mono_ind)
lrr_se <- (log(.710) - log(.365))/(2 * qnorm(.975))
rr <- rlnorm(n_samples, meanlog = log(.509), sdlog = lrr_se)
rr_indices <- list( # Indices of transition matrix to apply RR to
  c(1, 2), c(1, 3), c(1, 4),
  c(2, 3), c(2, 4),
  c(3, 4)
)
rr_mat <- matrix(rr, nrow = n_samples, ncol = length(rr_indices))
p[,, combo_ind] <- apply_rr(p[, , mono_ind],
                            rr = rr_mat,
                            index = rr_indices)
tp <- tparams_transprobs(p, p_id)

## Utility
utility_tbl <- stateval_tbl(
  data.table(
    state_id = 1:3,
    est = c(1, 1, 1)
  ),
  dist = "fixed"
)

## Costs
drugcost_tbl <- stateval_tbl(
  data.table(
    strategy_id = c(1, 1, 2, 2),
    time_start = c(0, 2, 0, 2),
    est = c(2278, 2278, 2278 + 2086.50, 2278)
  ),
  dist = "fixed"
)

dmedcost_tbl <- stateval_tbl(
  data.table(
    state_id = 1:3,
    mean = c(A = 1701, B = 1774, C = 6948),
    se = c(A = 1701, B = 1774, C = 6948)
  ),
  dist = "gamma"
)

cmedcost_tbl <- stateval_tbl(
  data.table(
    state_id = 1:3,
    mean = c(A = 1055, B = 1278, C = 2059),
    se = c(A = 1055, B = 1278, C = 2059)
  ),
  dist = "gamma"
)

# (2) Simulation
## Constructing the economic model
### Transition probabilities
transmod <- CohortDtstmTrans$new(params = tp)

### Utility
utilitymod <- create_StateVals(utility_tbl,
                               hesim_data = hesim_dat,
                               n = n_samples)

### Costs
drugcostmod <- create_StateVals(drugcost_tbl,
                                hesim_data = hesim_dat,
                                n = n_samples)
dmedcostmod <- create_StateVals(dmedcost_tbl,
                                hesim_data = hesim_dat,
                                n = n_samples)
cmedcostmod <- create_StateVals(cmedcost_tbl,
                                hesim_data = hesim_dat,
                                n = n_samples)
costmods <- list(drug = drugcostmod,
                 direct_medical = dmedcostmod,
                 community_medical = cmedcostmod)

### Economic model
econmod <- CohortDtstm$new(trans_model = transmod,
                           utility_model = utilitymod,
                           cost_models = costmods)

## Simulating outcomes
econmod$sim_stateprobs(n_cycles = 20)
autoplot(econmod$stateprobs_, ci = TRUE, ci_style = "ribbon",
         labels = labs)
econmod$sim_qalys(dr = 0, integrate_method = "riemann_right")
econmod$sim_costs(dr = 0.06, integrate_method = "riemann_right")

# (3) Decision analysis
ce_sim <- econmod$summarize()
wtp <- seq(0, 25000, 500)
cea_pw_out <- cea_pw(ce_sim, comparator = 1, dr_qalys = 0, dr_costs = .06,
                     k = wtp)
format(icer(cea_pw_out))
}
\references{
\href{https://arxiv.org/abs/2102.09437}{Incerti and Jansen (2021)}.
See Section 2.1 for a description of a cohort DTSTM and details on
simulating costs and QALYs from state probabilities. An example in oncology
is provided in Section 4.3.
}
\seealso{
\code{CohortDtstm} objects can be created from model objects as
documented in \code{\link[=create_CohortDtstm]{create_CohortDtstm()}}. The \code{\link{CohortDtstmTrans}} documentation
describes the class for the transition model and the \code{\link{StateVals}} documentation
describes the class for the cost and utility models. A \code{\link{CohortDtstmTrans}}
object is typically created using \code{\link[=create_CohortDtstmTrans]{create_CohortDtstmTrans()}}.

There are currently three relevant vignettes. \code{vignette("markov-cohort")}
details a relatively simple Markov model and
\code{vignette("markov-inhomogeneous-cohort")} describes a more complex time
inhomogeneous model in which transition probabilities vary in every model
cycle. The \code{vignette("mlogit")} shows how a transition model can be parameterized
using a multinomial logistic regression model when transition data is collected
at evenly spaced intervals.
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{trans_model}}{The model for health state transitions. Must be an object
of class \code{\link{CohortDtstmTrans}}.}

\item{\code{utility_model}}{The model for health state utility. Must be an object of
class \code{\link{StateVals}}.}

\item{\code{cost_models}}{The models used to predict costs by health state.
Must be a list of objects of class \code{\link{StateVals}}, where each element of the
list represents a different cost category.}

\item{\code{stateprobs_}}{An object of class \code{\link{stateprobs}} simulated using \verb{$sim_stateprobs()}.}

\item{\code{qalys_}}{An object of class \code{\link{qalys}} simulated using \verb{$sim_qalys()}.}

\item{\code{costs_}}{An object of class \code{\link{costs}} simulated using \verb{$sim_costs()}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-CohortDtstm-new}{\code{CohortDtstm$new()}}
\item \href{#method-CohortDtstm-sim_stateprobs}{\code{CohortDtstm$sim_stateprobs()}}
\item \href{#method-CohortDtstm-sim_qalys}{\code{CohortDtstm$sim_qalys()}}
\item \href{#method-CohortDtstm-sim_costs}{\code{CohortDtstm$sim_costs()}}
\item \href{#method-CohortDtstm-summarize}{\code{CohortDtstm$summarize()}}
\item \href{#method-CohortDtstm-clone}{\code{CohortDtstm$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-new"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{CohortDtstm} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$new(trans_model = NULL, utility_model = NULL, cost_models = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{trans_model}}{The \code{trans_model} field.}

\item{\code{utility_model}}{The \code{utility_model} field.}

\item{\code{cost_models}}{The \code{cost_models} field.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{CohortDtstm} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-sim_stateprobs"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-sim_stateprobs}{}}}
\subsection{Method \code{sim_stateprobs()}}{
Simulate health state probabilities using \code{CohortDtstmTrans$sim_stateprobs()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_stateprobs(n_cycles)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{n_cycles}}{The number of model cycles to simulate the model for.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{stateprobs}
stored in \code{stateprobs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-sim_qalys"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-sim_qalys}{}}}
\subsection{Method \code{sim_qalys()}}{
Simulate quality-adjusted life-years (QALYs) as a function of \code{stateprobs_} and
\code{utility_model}. See \code{sim_qalys()} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_qalys(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right"),
  lys = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing
costs or QALYs. Options are \code{trapz} for the trapezoid rule,
\code{riemann_left} for a left Riemann sum, and
\code{riemann_right} for a right Riemann sum.}

\item{\code{lys}}{If \code{TRUE}, then life-years are simulated in addition to QALYs.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{qalys} stored
in \code{qalys_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-sim_costs"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-sim_costs}{}}}
\subsection{Method \code{sim_costs()}}{
Simulate costs as a function of \code{stateprobs_} and \code{cost_models}.
See \code{sim_costs()} for details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$sim_costs(
  dr = 0.03,
  integrate_method = c("trapz", "riemann_left", "riemann_right")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{integrate_method}}{Method used to integrate state values when computing
costs or QALYs. Options are \code{trapz} for the trapezoid rule,
\code{riemann_left} for a left Riemann sum, and
\code{riemann_right} for a right Riemann sum.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{costs} stored
in \code{costs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-summarize"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-summarize}{}}}
\subsection{Method \code{summarize()}}{
Summarize costs and QALYs so that cost-effectiveness analysis can be performed.
See \code{\link[=summarize_ce]{summarize_ce()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$summarize(by_grp = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{by_grp}}{If \code{TRUE}, then costs and QALYs are computed by subgroup. If
\code{FALSE}, then costs and QALYs are aggregated across all patients (and subgroups).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CohortDtstm-clone"></a>}}
\if{latex}{\out{\hypertarget{method-CohortDtstm-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CohortDtstm$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
