/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tconv.h"    
#include "H5Tconv_compound.h"

typedef struct H5T_conv_struct_t {
    int              *src2dst;     
    H5T_t           **src_memb;    
    H5T_t           **dst_memb;    
    hid_t            *src_memb_id; 
    hid_t            *dst_memb_id; 
    H5T_path_t      **memb_path;   
    H5T_subset_info_t subset_info; 
    unsigned          src_nmembs;  
} H5T_conv_struct_t;

static herr_t H5T__conv_struct_init(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata,
                                    const H5T_conv_ctx_t *conv_ctx);
static herr_t H5T__conv_struct_free(H5T_conv_struct_t *priv);

H5T_subset_info_t *
H5T__conv_struct_subset(const H5T_cdata_t *cdata)
{
    H5T_conv_struct_t *priv = NULL;

    FUNC_ENTER_PACKAGE_NOERR

    assert(cdata);
    assert(cdata->priv);

    priv = (H5T_conv_struct_t *)(cdata->priv);

    FUNC_LEAVE_NOAPI((H5T_subset_info_t *)&priv->subset_info)
} 

static herr_t
H5T__conv_struct_init(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx)
{
    H5T_conv_struct_t *priv    = (H5T_conv_struct_t *)(cdata->priv);
    int               *src2dst = NULL;
    unsigned           src_nmembs, dst_nmembs;
    unsigned           i, j;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    src_nmembs = src->shared->u.compnd.nmembs;
    dst_nmembs = dst->shared->u.compnd.nmembs;

    if (!priv) {
        
        if (NULL == (priv = (H5T_conv_struct_t *)(cdata->priv = H5MM_calloc(sizeof(H5T_conv_struct_t)))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "couldn't allocate private conversion data");
        if (NULL == (priv->src2dst = (int *)H5MM_malloc(src_nmembs * sizeof(int))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate source to destination member mapping array");
        if (NULL == (priv->src_memb = (H5T_t **)H5MM_malloc(src_nmembs * sizeof(H5T_t *))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate source compound member datatype array");
        if (NULL == (priv->dst_memb = (H5T_t **)H5MM_malloc(dst_nmembs * sizeof(H5T_t *))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate destination compound member datatype array");

        
        if (NULL == (priv->src_memb_id = (hid_t *)H5MM_malloc(src_nmembs * sizeof(hid_t))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate source compound member datatype ID array");
        for (i = 0; i < src_nmembs; i++)
            priv->src_memb_id[i] = H5I_INVALID_HID;

        if (NULL == (priv->dst_memb_id = (hid_t *)H5MM_malloc(dst_nmembs * sizeof(hid_t))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate destination compound member datatype ID array");
        for (i = 0; i < dst_nmembs; i++)
            priv->dst_memb_id[i] = H5I_INVALID_HID;

        src2dst          = priv->src2dst;
        priv->src_nmembs = src_nmembs;

        
        priv->subset_info.subset    = H5T_SUBSET_FALSE;
        priv->subset_info.copy_size = 0;

        
        H5T__sort_value(src, NULL);
        H5T__sort_value(dst, NULL);

        
        for (i = 0; i < src_nmembs; i++) {
            src2dst[i] = -1;
            for (j = 0; j < dst_nmembs; j++) {
                if (!strcmp(src->shared->u.compnd.memb[i].name, dst->shared->u.compnd.memb[j].name)) {
                    H5_CHECKED_ASSIGN(src2dst[i], int, j, unsigned);
                    break;
                } 
            }     
            if (src2dst[i] >= 0) {
                H5T_t *type;

                if (NULL == (type = H5T_copy(src->shared->u.compnd.memb[i].type, H5T_COPY_ALL)))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL,
                                "can't copy source compound member datatype");
                priv->src_memb[i] = type;

                if (NULL == (type = H5T_copy(dst->shared->u.compnd.memb[src2dst[i]].type, H5T_COPY_ALL)))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL,
                                "can't copy destination compound member datatype");
                priv->dst_memb[src2dst[i]] = type;
            } 
        }     
    }         
    else {
        
        
        H5T__sort_value(src, NULL);
        H5T__sort_value(dst, NULL);
    } 

    
    src2dst = priv->src2dst;
    H5MM_xfree(priv->memb_path);
    if (NULL ==
        (priv->memb_path = (H5T_path_t **)H5MM_malloc(src->shared->u.compnd.nmembs * sizeof(H5T_path_t *))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    for (i = 0; i < src_nmembs; i++) {
        if (src2dst[i] >= 0) {
            H5T_path_t *tpath;
            bool        need_ids;

            tpath = H5T_path_find(src->shared->u.compnd.memb[i].type,
                                  dst->shared->u.compnd.memb[src2dst[i]].type);

            if (NULL == (priv->memb_path[i] = tpath)) {
                H5T__conv_struct_free(priv);
                cdata->priv = NULL;
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unable to convert member datatype");
            } 

            
            need_ids = tpath->conv.is_app ||
                       (cdata->command == H5T_CONV_INIT && conv_ctx->u.init.cb_struct.func) ||
                       (cdata->command == H5T_CONV_CONV && conv_ctx->u.conv.cb_struct.func);

            if (need_ids) {
                hid_t tid;

                
                if (priv->src_memb_id[i] == H5I_INVALID_HID) {
                    if ((tid = H5I_register(H5I_DATATYPE, priv->src_memb[i], false)) < 0) {
                        H5T__conv_struct_free(priv);
                        cdata->priv = NULL;
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                                    "can't register ID for source compound member datatype");
                    }
                    priv->src_memb_id[i] = tid;
                }

                if (priv->dst_memb_id[src2dst[i]] == H5I_INVALID_HID) {
                    if ((tid = H5I_register(H5I_DATATYPE, priv->dst_memb[src2dst[i]], false)) < 0) {
                        H5T__conv_struct_free(priv);
                        cdata->priv = NULL;
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                                    "can't register ID for destination compound member datatype");
                    }
                    priv->dst_memb_id[src2dst[i]] = tid;
                }
            }
        } 
    }     

    
    cdata->need_bkg = H5T_BKG_YES;

    if (src_nmembs < dst_nmembs) {
        priv->subset_info.subset = H5T_SUBSET_SRC;
        for (i = 0; i < src_nmembs; i++) {
            
            if (src2dst[i] != (int)i ||
                (src->shared->u.compnd.memb[i].offset != dst->shared->u.compnd.memb[i].offset) ||
                (priv->memb_path[i])->is_noop == false) {
                priv->subset_info.subset = H5T_SUBSET_FALSE;
                break;
            } 
        }     
        
        if (priv->subset_info.subset == H5T_SUBSET_SRC)
            priv->subset_info.copy_size = src->shared->u.compnd.memb[src_nmembs - 1].offset +
                                          src->shared->u.compnd.memb[src_nmembs - 1].size;
    }
    else if (dst_nmembs < src_nmembs) {
        priv->subset_info.subset = H5T_SUBSET_DST;
        for (i = 0; i < dst_nmembs; i++) {
            
            if (src2dst[i] != (int)i ||
                (src->shared->u.compnd.memb[i].offset != dst->shared->u.compnd.memb[i].offset) ||
                (priv->memb_path[i])->is_noop == false) {
                priv->subset_info.subset = H5T_SUBSET_FALSE;
                break;
            }
        } 
        
        if (priv->subset_info.subset == H5T_SUBSET_DST)
            priv->subset_info.copy_size = dst->shared->u.compnd.memb[dst_nmembs - 1].offset +
                                          dst->shared->u.compnd.memb[dst_nmembs - 1].size;
    }
    else 
    {
    }

    cdata->recalc = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__conv_struct_free(H5T_conv_struct_t *priv)
{
    int    *src2dst     = priv->src2dst;
    H5T_t **src_memb    = priv->src_memb;
    H5T_t **dst_memb    = priv->dst_memb;
    hid_t  *src_memb_id = priv->src_memb_id;
    hid_t  *dst_memb_id = priv->dst_memb_id;
    herr_t  ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR

    for (unsigned i = 0; i < priv->src_nmembs; i++)
        if (src2dst[i] >= 0) {
            if (src_memb_id[i] >= 0) {
                if (H5I_dec_ref(src_memb_id[i]) < 0)
                    ret_value = FAIL; 
                src_memb_id[i] = H5I_INVALID_HID;
                src_memb[i]    = NULL;
            }
            else {
                if (H5T_close(src_memb[i]) < 0)
                    ret_value = FAIL; 
                src_memb[i] = NULL;
            }
            if (dst_memb_id[src2dst[i]] >= 0) {
                if (H5I_dec_ref(dst_memb_id[src2dst[i]]) < 0)
                    ret_value = FAIL; 
                dst_memb_id[src2dst[i]] = H5I_INVALID_HID;
                dst_memb[src2dst[i]]    = NULL;
            }
            else {
                if (H5T_close(dst_memb[src2dst[i]]) < 0)
                    ret_value = FAIL; 
                dst_memb[src2dst[i]] = NULL;
            }
        } 

    H5MM_xfree(src2dst);
    H5MM_xfree(src_memb);
    H5MM_xfree(dst_memb);
    H5MM_xfree(src_memb_id);
    H5MM_xfree(dst_memb_id);

    H5MM_xfree(priv->memb_path);
    H5MM_xfree(priv);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_struct(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                 size_t nelmts, size_t buf_stride, size_t bkg_stride, void *_buf, void *_bkg)
{
    uint8_t           *buf  = (uint8_t *)_buf;  
    uint8_t           *bkg  = (uint8_t *)_bkg;  
    uint8_t           *xbuf = buf, *xbkg = bkg; 
    int               *src2dst  = NULL;         
    H5T_cmemb_t       *src_memb = NULL;         
    H5T_cmemb_t       *dst_memb = NULL;         
    size_t             offset;                  
    ssize_t            src_delta;               
    ssize_t            bkg_delta;               
    size_t             elmtno;
    unsigned           u; 
    H5T_conv_struct_t *priv         = (H5T_conv_struct_t *)(cdata->priv);
    H5T_conv_ctx_t     tmp_conv_ctx = {0};
    herr_t             ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT:
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a datatype");
            if (H5T_COMPOUND != src->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_COMPOUND datatype");
            if (H5T_COMPOUND != dst->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_COMPOUND datatype");

            if (H5T__conv_struct_init(src, dst, cdata, conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize conversion data");
            break;

        case H5T_CONV_FREE: {
            
            herr_t status = H5T__conv_struct_free(priv);
            cdata->priv   = NULL;
            if (status < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free private conversion data");

            break;
        }

        case H5T_CONV_CONV:
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");
            assert(priv);
            assert(bkg && cdata->need_bkg);

            
            tmp_conv_ctx = *conv_ctx;

            if (cdata->recalc && H5T__conv_struct_init(src, dst, cdata, conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize conversion data");

            
            H5T__sort_value(src, NULL);
            H5T__sort_value(dst, NULL);
            src2dst = priv->src2dst;

            
            if (buf_stride) {
                H5_CHECKED_ASSIGN(src_delta, ssize_t, buf_stride, size_t);
                if (!bkg_stride) {
                    H5_CHECKED_ASSIGN(bkg_delta, ssize_t, dst->shared->size, size_t);
                } 
                else
                    H5_CHECKED_ASSIGN(bkg_delta, ssize_t, bkg_stride, size_t);
            } 
            else if (dst->shared->size <= src->shared->size) {
                H5_CHECKED_ASSIGN(src_delta, ssize_t, src->shared->size, size_t);
                H5_CHECKED_ASSIGN(bkg_delta, ssize_t, dst->shared->size, size_t);
            } 
            else {
                H5_CHECK_OVERFLOW(src->shared->size, size_t, ssize_t);
                src_delta = -(ssize_t)src->shared->size;
                H5_CHECK_OVERFLOW(dst->shared->size, size_t, ssize_t);
                bkg_delta = -(ssize_t)dst->shared->size;
                xbuf += (nelmts - 1) * src->shared->size;
                xbkg += (nelmts - 1) * dst->shared->size;
            } 

            
            for (elmtno = 0; elmtno < nelmts; elmtno++) {
                
                tmp_conv_ctx.u.conv.recursive = true;
                for (u = 0, offset = 0; u < src->shared->u.compnd.nmembs; u++) {
                    if (src2dst[u] < 0)
                        continue; 
                    src_memb = src->shared->u.compnd.memb + u;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[u];

                    if (dst_memb->size <= src_memb->size) {
                        
                        tmp_conv_ctx.u.conv.src_type_id = priv->src_memb_id[u];
                        tmp_conv_ctx.u.conv.dst_type_id = priv->dst_memb_id[src2dst[u]];

                        if (H5T_convert_with_ctx(priv->memb_path[u], priv->src_memb[u],
                                                 priv->dst_memb[src2dst[u]], &tmp_conv_ctx, (size_t)1,
                                                 (size_t)0, (size_t)0, 
                                                 xbuf + src_memb->offset, xbkg + dst_memb->offset) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL,
                                        "unable to convert compound datatype member");

                        memmove(xbuf + offset, xbuf + src_memb->offset, dst_memb->size);
                        offset += dst_memb->size;
                    } 
                    else {
                        memmove(xbuf + offset, xbuf + src_memb->offset, src_memb->size);
                        offset += src_memb->size;
                    } 
                }     
                tmp_conv_ctx.u.conv.recursive = false;

                
                tmp_conv_ctx.u.conv.recursive = true;
                H5_CHECK_OVERFLOW(src->shared->u.compnd.nmembs, size_t, int);
                for (int i = (int)src->shared->u.compnd.nmembs - 1; i >= 0; --i) {
                    if (src2dst[i] < 0)
                        continue; 
                    src_memb = src->shared->u.compnd.memb + i;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[i];

                    if (dst_memb->size > src_memb->size) {
                        
                        tmp_conv_ctx.u.conv.src_type_id = priv->src_memb_id[i];
                        tmp_conv_ctx.u.conv.dst_type_id = priv->dst_memb_id[src2dst[i]];

                        offset -= src_memb->size;
                        if (H5T_convert_with_ctx(priv->memb_path[i], priv->src_memb[i],
                                                 priv->dst_memb[src2dst[i]], &tmp_conv_ctx, (size_t)1,
                                                 (size_t)0, (size_t)0, 
                                                 xbuf + offset, xbkg + dst_memb->offset) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL,
                                        "unable to convert compound datatype member");
                    } 
                    else
                        offset -= dst_memb->size;
                    memcpy(xbkg + dst_memb->offset, xbuf + offset, dst_memb->size);
                } 
                tmp_conv_ctx.u.conv.recursive = false;

                assert(0 == offset);

                
                xbuf += src_delta;
                xbkg += bkg_delta;
            } 

            
            if (buf_stride == 0 && dst->shared->size > src->shared->size)
                H5_CHECKED_ASSIGN(bkg_delta, ssize_t, dst->shared->size, size_t);

            
            for (xbuf = buf, xbkg = bkg, elmtno = 0; elmtno < nelmts; elmtno++) {
                memcpy(xbuf, xbkg, dst->shared->size);
                xbuf += buf_stride ? buf_stride : dst->shared->size;
                xbkg += bkg_delta;
            } 
            break;

        default:
            
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_struct_opt(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                     size_t nelmts, size_t buf_stride, size_t bkg_stride, void *_buf, void *_bkg)
{
    uint8_t           *buf      = (uint8_t *)_buf; 
    uint8_t           *bkg      = (uint8_t *)_bkg; 
    uint8_t           *xbuf     = NULL;            
    uint8_t           *xbkg     = NULL;            
    int               *src2dst  = NULL;            
    H5T_cmemb_t       *src_memb = NULL;            
    H5T_cmemb_t       *dst_memb = NULL;            
    size_t             offset;                     
    size_t             elmtno;                     
    size_t             copy_size;                  
    H5T_conv_struct_t *priv         = NULL;        
    H5T_conv_ctx_t     tmp_conv_ctx = {0};         
    bool               no_stride    = false;       
    unsigned           u;                          
    herr_t             ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT:
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (H5T_COMPOUND != src->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_COMPOUND datatype");
            if (H5T_COMPOUND != dst->shared->type)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a H5T_COMPOUND datatype");

            
            if (H5T__conv_struct_init(src, dst, cdata, conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize conversion data");
            priv    = (H5T_conv_struct_t *)(cdata->priv);
            src2dst = priv->src2dst;

            
            if (dst->shared->size > src->shared->size) {
                for (u = 0, offset = 0; u < src->shared->u.compnd.nmembs; u++) {
                    if (src2dst[u] < 0)
                        continue;
                    src_memb = src->shared->u.compnd.memb + u;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[u];
                    if (dst_memb->size > src_memb->size)
                        offset += src_memb->size;
                } 
                H5_CHECK_OVERFLOW(src->shared->u.compnd.nmembs, size_t, int);
                for (int i = (int)src->shared->u.compnd.nmembs - 1; i >= 0; --i) {
                    if (src2dst[i] < 0)
                        continue;
                    src_memb = src->shared->u.compnd.memb + i;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[i];
                    if (dst_memb->size > src_memb->size) {
                        offset -= src_memb->size;
                        if (dst_memb->size > src->shared->size - offset) {
                            H5T__conv_struct_free(priv);
                            cdata->priv = NULL;
                            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                        "conversion is unsupported by this function");
                        } 
                    }     
                }         
            }             
            break;

        case H5T_CONV_FREE: {
            
            herr_t status = H5T__conv_struct_free((H5T_conv_struct_t *)(cdata->priv));
            cdata->priv   = NULL;
            if (status < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free private conversion data");

            break;
        }

        case H5T_CONV_CONV:
            
            if (NULL == src || NULL == dst)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");
            if (!bkg)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid background buffer pointer");

            
            tmp_conv_ctx = *conv_ctx;

            
            if (cdata->recalc && H5T__conv_struct_init(src, dst, cdata, conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize conversion data");
            priv = (H5T_conv_struct_t *)(cdata->priv);
            assert(priv);
            src2dst = priv->src2dst;
            assert(cdata->need_bkg);

            
            H5T__sort_value(src, NULL);
            H5T__sort_value(dst, NULL);

            
            if (!buf_stride || !bkg_stride)
                bkg_stride = dst->shared->size;
            if (!buf_stride) {
                no_stride  = true;
                buf_stride = src->shared->size;
            } 

            if (priv->subset_info.subset == H5T_SUBSET_SRC || priv->subset_info.subset == H5T_SUBSET_DST) {
                
                xbuf      = buf;
                xbkg      = bkg;
                copy_size = priv->subset_info.copy_size;

                for (elmtno = 0; elmtno < nelmts; elmtno++) {
                    memcpy(xbkg, xbuf, copy_size);

                    
                    xbuf += buf_stride;
                    xbkg += bkg_stride;
                } 
            }     
            else {
                
                tmp_conv_ctx.u.conv.recursive = true;
                for (u = 0, offset = 0; u < src->shared->u.compnd.nmembs; u++) {
                    if (src2dst[u] < 0)
                        continue; 
                    src_memb = src->shared->u.compnd.memb + u;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[u];

                    if (dst_memb->size <= src_memb->size) {
                        
                        tmp_conv_ctx.u.conv.src_type_id = priv->src_memb_id[u];
                        tmp_conv_ctx.u.conv.dst_type_id = priv->dst_memb_id[src2dst[u]];

                        xbuf = buf + src_memb->offset;
                        xbkg = bkg + dst_memb->offset;
                        if (H5T_convert_with_ctx(priv->memb_path[u], priv->src_memb[u],
                                                 priv->dst_memb[src2dst[u]], &tmp_conv_ctx, nelmts,
                                                 buf_stride, bkg_stride, xbuf, xbkg) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL,
                                        "unable to convert compound datatype member");

                        for (elmtno = 0; elmtno < nelmts; elmtno++) {
                            memcpy(xbkg, xbuf, dst_memb->size);
                            xbuf += buf_stride;
                            xbkg += bkg_stride;
                        } 
                    }     
                    else {
                        for (xbuf = buf, elmtno = 0; elmtno < nelmts; elmtno++) {
                            memmove(xbuf + offset, xbuf + src_memb->offset, src_memb->size);
                            xbuf += buf_stride;
                        } 
                        offset += src_memb->size;
                    } 
                }     
                tmp_conv_ctx.u.conv.recursive = false;

                
                tmp_conv_ctx.u.conv.recursive = true;
                H5_CHECK_OVERFLOW(src->shared->u.compnd.nmembs, size_t, int);
                for (int i = (int)src->shared->u.compnd.nmembs - 1; i >= 0; --i) {
                    if (src2dst[i] < 0)
                        continue;
                    src_memb = src->shared->u.compnd.memb + i;
                    dst_memb = dst->shared->u.compnd.memb + src2dst[i];

                    if (dst_memb->size > src_memb->size) {
                        
                        tmp_conv_ctx.u.conv.src_type_id = priv->src_memb_id[i];
                        tmp_conv_ctx.u.conv.dst_type_id = priv->dst_memb_id[src2dst[i]];

                        offset -= src_memb->size;
                        xbuf = buf + offset;
                        xbkg = bkg + dst_memb->offset;
                        if (H5T_convert_with_ctx(priv->memb_path[i], priv->src_memb[i],
                                                 priv->dst_memb[src2dst[i]], &tmp_conv_ctx, nelmts,
                                                 buf_stride, bkg_stride, xbuf, xbkg) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL,
                                        "unable to convert compound datatype member");
                        for (elmtno = 0; elmtno < nelmts; elmtno++) {
                            memcpy(xbkg, xbuf, dst_memb->size);
                            xbuf += buf_stride;
                            xbkg += bkg_stride;
                        } 
                    }     
                }         
                tmp_conv_ctx.u.conv.recursive = false;
            } 

            if (no_stride)
                buf_stride = dst->shared->size;

            
            for (xbuf = buf, xbkg = bkg, elmtno = 0; elmtno < nelmts; elmtno++) {
                memcpy(xbuf, xbkg, dst->shared->size);
                xbuf += buf_stride;
                xbkg += bkg_stride;
            } 
            break;

        default:
            
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
