/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Sprivate.h"  

static herr_t H5D__ioinfo_init(size_t count, H5D_io_op_type_t op_type, H5D_dset_io_info_t *dset_info,
                               H5D_io_info_t *io_info);
static herr_t H5D__dset_ioinfo_init(H5D_t *dset, H5D_dset_io_info_t *dset_info, H5D_storage_t *store);
static herr_t H5D__typeinfo_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info,
                                 const H5T_t *mem_type);
static herr_t H5D__typeinfo_init_phase2(H5D_io_info_t *io_info);
static herr_t H5D__typeinfo_init_phase3(H5D_io_info_t *io_info);
#ifdef H5_HAVE_PARALLEL
static herr_t H5D__ioinfo_adjust(H5D_io_info_t *io_info);
#endif 
static herr_t H5D__typeinfo_term(H5D_io_info_t *io_info);

H5FL_BLK_DEFINE(type_conv);

herr_t
H5D__read(size_t count, H5D_dset_io_info_t *dset_info)
{
    H5D_io_info_t io_info;                    
    H5S_t        *orig_mem_space_local;       
    H5S_t       **orig_mem_space = NULL;      
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
    H5D_storage_t  store_local;               
    H5D_storage_t *store      = &store_local; 
    size_t         io_op_init = 0;            
    size_t         io_skipped =
        0;            
    size_t i;         
    char   fake_char; 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

#ifdef H5_HAVE_PARALLEL
    
    if (!H5CX_is_def_dxpl()) {
        H5CX_set_mpio_actual_chunk_opt(H5D_MPIO_NO_CHUNK_OPTIMIZATION);
        H5CX_set_mpio_actual_io_mode(H5D_MPIO_NO_COLLECTIVE);
    } 
#endif

    
    if (H5D__ioinfo_init(count, H5D_IO_OP_READ, dset_info, &io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize I/O info");

    
    if (count > 1)
        if (NULL == (store = (H5D_storage_t *)H5MM_malloc(count * sizeof(H5D_storage_t))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "couldn't allocate dset storage info array buffer");

#ifdef H5_HAVE_PARALLEL
    
    if (!(H5F_HAS_FEATURE(dset_info[0].dset->oloc.file, H5FD_FEAT_HAS_MPI))) {
        H5FD_mpio_xfer_t io_xfer_mode; 

        
        if (H5CX_get_io_xfer_mode(&io_xfer_mode) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI-I/O transfer mode");

        
        if (io_xfer_mode == H5FD_MPIO_COLLECTIVE)
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "collective access for MPI-based drivers only");
    }  
#endif 

    
    for (i = 0; i < count; i++) {
        haddr_t prev_tag = HADDR_UNDEF;

        
        if (NULL == dset_info[i].dset)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset");
        if (NULL == dset_info[i].dset->oloc.file)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

        
        H5AC_tag(dset_info[i].dset->oloc.addr, &prev_tag);

        
        if (H5D__typeinfo_init(&io_info, &(dset_info[i]), dset_info[i].mem_type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info");

        
        dset_info[i].nelmts = H5S_GET_SELECT_NPOINTS(dset_info[i].mem_space);
        if (dset_info[i].nelmts != H5S_GET_SELECT_NPOINTS(dset_info[i].file_space))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "src and dest dataspaces have different number of elements selected");

        
        if (NULL == dset_info[i].buf.vp) {
            
            if (dset_info[i].nelmts > 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no output buffer");

            
            dset_info[i].buf.vp = &fake_char;
        } 

        
        if (!(H5S_has_extent(dset_info[i].file_space)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file dataspace does not have extent set");
        if (!(H5S_has_extent(dset_info[i].mem_space)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "memory dataspace does not have extent set");

        
        if (dset_info[i].nelmts > 0 &&
            true == H5S_SELECT_SHAPE_SAME(dset_info[i].mem_space, dset_info[i].file_space) &&
            H5S_GET_EXTENT_NDIMS(dset_info[i].mem_space) != H5S_GET_EXTENT_NDIMS(dset_info[i].file_space)) {
            ptrdiff_t buf_adj = 0;

            
            if (!orig_mem_space) {
                if (count > 1) {
                    
                    if (NULL == (orig_mem_space = (H5S_t **)H5MM_calloc(count * sizeof(H5S_t *))))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL,
                                    "couldn't allocate original memory space array buffer");
                }
                else
                    
                    orig_mem_space = &orig_mem_space_local;
            }

            
            orig_mem_space[i]      = dset_info[i].mem_space;
            dset_info[i].mem_space = NULL;

            
            if (H5S_select_construct_projection(orig_mem_space[i], &dset_info[i].mem_space,
                                                (unsigned)H5S_GET_EXTENT_NDIMS(dset_info[i].file_space),
                                                (hsize_t)dset_info[i].type_info.dst_type_size, &buf_adj) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to construct projected memory dataspace");
            assert(dset_info[i].mem_space);

            
            dset_info[i].buf.vp = (void *)(((uint8_t *)dset_info[i].buf.vp) + buf_adj);
        } 

        
        if (H5D__dset_ioinfo_init(dset_info[i].dset, &(dset_info[i]), &(store[i])) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up I/O operation");

        
        if (dset_info[i].dset->shared->dcpl_cache.pline.nused > 0)
            io_info.filtered_count++;

        
        if (dset_info[i].nelmts > 0 && dset_info[i].dset->shared->dcpl_cache.efl.nused == 0 &&
            !(*dset_info[i].dset->shared->layout.ops->is_space_alloc)(
                &dset_info[i].dset->shared->layout.storage) &&
            !(dset_info[i].dset->shared->layout.ops->is_data_cached &&
              (*dset_info[i].dset->shared->layout.ops->is_data_cached)(dset_info[i].dset->shared))) {
            H5D_fill_value_t fill_status; 

            
            if (H5P_is_fill_value_defined(&dset_info[i].dset->shared->dcpl_cache.fill, &fill_status) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

            
            if (fill_status == H5D_FILL_VALUE_UNDEFINED &&
                (dset_info[i].dset->shared->dcpl_cache.fill.fill_time == H5D_FILL_TIME_ALLOC ||
                 dset_info[i].dset->shared->dcpl_cache.fill.fill_time == H5D_FILL_TIME_IFSET))
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL,
                            "read failed: dataset doesn't exist, no data can be read");

            
            if (dset_info[i].dset->shared->dcpl_cache.fill.fill_time != H5D_FILL_TIME_NEVER)
                
                if (H5D__fill(dset_info[i].dset->shared->dcpl_cache.fill.buf, dset_info[i].dset->shared->type,
                              dset_info[i].buf.vp, dset_info[i].type_info.mem_type,
                              dset_info[i].mem_space) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "filling buf failed");

            
            dset_info[i].skip_io = true;
            io_skipped           = io_skipped + 1;
        } 
        else {
            
            if (dset_info[i].nelmts > 0)
                assert(
                    (*dset_info[i].dset->shared->layout.ops->is_space_alloc)(
                        &dset_info[i].dset->shared->layout.storage) ||
                    (dset_info[i].dset->shared->layout.ops->is_data_cached &&
                     (*dset_info[i].dset->shared->layout.ops->is_data_cached)(dset_info[i].dset->shared)) ||
                    dset_info[i].dset->shared->dcpl_cache.efl.nused > 0 ||
                    dset_info[i].dset->shared->layout.type == H5D_COMPACT);

            dset_info[i].skip_io = false;
        }

        
        if (dset_info[i].layout_ops.io_init &&
            (dset_info[i].layout_ops.io_init)(&io_info, &(dset_info[i])) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize I/O info");
        io_op_init++;

        
        H5AC_tag(prev_tag, NULL);
    } 

    assert(io_op_init == count);

    
    if (io_skipped == count)
        HGOTO_DONE(SUCCEED);

    
    if (H5D__typeinfo_init_phase2(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info (second phase)");

#ifdef H5_HAVE_PARALLEL
    
    if (H5D__ioinfo_adjust(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "unable to adjust I/O info for parallel or selection I/O");
#endif 

    
    if (H5D__typeinfo_init_phase3(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info (third phase)");

    H5CX_set_no_selection_io_cause(io_info.no_selection_io_cause);

    
    if (io_info.md_io_ops.multi_read_md) {
        
        if (io_info.piece_count > 0) {
            assert(!io_info.sel_pieces);
            assert(io_info.pieces_added == 0);

            
            if (NULL ==
                (io_info.sel_pieces = H5MM_malloc(io_info.piece_count * sizeof(io_info.sel_pieces[0]))))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "unable to allocate array of selected pieces");
        }

        
        for (i = 0; i < count; i++) {
            
            if (dset_info[i].skip_io)
                continue;

            if (dset_info[i].layout_ops.mdio_init) {
                haddr_t prev_tag = HADDR_UNDEF;

                
                H5AC_tag(dset_info[i].dset->oloc.addr, &prev_tag);

                
                if ((dset_info[i].layout_ops.mdio_init)(&io_info, &(dset_info[i])) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't populate array of selected pieces");

                
                H5AC_tag(prev_tag, NULL);
            }
        }

        
        if ((*io_info.md_io_ops.multi_read_md)(&io_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read data");
    } 
    else {
        haddr_t prev_tag = HADDR_UNDEF;

        
        if (!H5D_LAYOUT_CB_PERFORM_IO(&io_info) && io_info.piece_count > 0) {
            if (NULL == (io_info.mem_spaces = H5MM_malloc(io_info.piece_count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for memory space list");
            if (NULL == (io_info.file_spaces = H5MM_malloc(io_info.piece_count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for file space list");
            if (NULL == (io_info.addrs = H5MM_malloc(io_info.piece_count * sizeof(haddr_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for piece address list");
            if (NULL == (io_info.element_sizes = H5MM_malloc(io_info.piece_count * sizeof(size_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for element size list");
            if (NULL == (io_info.rbufs = H5MM_malloc(io_info.piece_count * sizeof(void *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for read buffer list");
            if (io_info.max_tconv_type_size > 0)
                if (NULL ==
                    (io_info.sel_pieces = H5MM_malloc(io_info.piece_count * sizeof(io_info.sel_pieces[0]))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "unable to allocate array of selected pieces");
        }

        
        for (i = 0; i < count; i++) {
            
            if (dset_info[i].skip_io)
                continue;

            
            H5AC_tag(dset_info[i].dset->oloc.addr, &prev_tag);

            
            if ((*dset_info[i].io_ops.multi_read)(&io_info, &dset_info[i]) < 0) {
                
                H5AC_tag(prev_tag, NULL);

                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read data");
            }

            
            H5AC_tag(prev_tag, NULL);
        }

        
        if (!H5D_LAYOUT_CB_PERFORM_IO(&io_info)) {
            
            if (io_info.max_tconv_type_size > 0) {
                
                if (H5D__scatgath_read_select(&io_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "type conversion selection read failed");
            }
            else {
                
                H5_CHECK_OVERFLOW(io_info.pieces_added, size_t, uint32_t);
                if (H5F_shared_select_read(io_info.f_sh, H5FD_MEM_DRAW, (uint32_t)io_info.pieces_added,
                                           io_info.mem_spaces, io_info.file_spaces, io_info.addrs,
                                           io_info.element_sizes, io_info.rbufs) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "selection read failed");
            }
        }

#ifdef H5_HAVE_PARALLEL
        
        if (io_info.using_mpi_vfd) {
            H5FD_mpio_xfer_t xfer_mode; 

            
            if (H5CX_get_io_xfer_mode(&xfer_mode) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI-I/O transfer mode");

            
            if (xfer_mode == H5FD_MPIO_COLLECTIVE) {
                H5CX_set_mpio_actual_io_mode(io_info.actual_io_mode);

                
                if (io_info.use_select_io == H5D_SELECTION_IO_MODE_ON)
                    H5CX_set_mpio_actual_chunk_opt(H5D_MPIO_LINK_CHUNK);
            }
        }
#endif 
    }

done:
    
    for (i = 0; i < io_op_init; i++)
        if (dset_info[i].layout_ops.io_term &&
            (*dset_info[i].layout_ops.io_term)(&io_info, &(dset_info[i])) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "unable to shut down I/O op info");

    
    if (H5D__typeinfo_term(&io_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "unable to shut down type info");

    
    if (orig_mem_space) {
        for (i = 0; i < count; i++)
            if (orig_mem_space[i]) {
                if (H5S_close(dset_info[i].mem_space) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL,
                                "unable to shut down projected memory dataspace");
                dset_info[i].mem_space = orig_mem_space[i];
            }

        
        if (orig_mem_space != &orig_mem_space_local)
            H5MM_free(orig_mem_space);
    }

    
    H5MM_xfree(io_info.sel_pieces);

    
    H5MM_xfree(io_info.mem_spaces);
    H5MM_xfree(io_info.file_spaces);
    H5MM_xfree(io_info.addrs);
    H5MM_xfree(io_info.element_sizes);
    H5MM_xfree(io_info.rbufs);

    
    if (store != &store_local)
        H5MM_free(store);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__write(size_t count, H5D_dset_io_info_t *dset_info)
{
    H5D_io_info_t io_info;                    
    H5S_t        *orig_mem_space_local;       
    H5S_t       **orig_mem_space = NULL;      
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
                                              
    H5D_storage_t  store_local;               
    H5D_storage_t *store      = &store_local; 
    size_t         io_op_init = 0;            
    size_t         i;                         
    char           fake_char;                 
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

#ifdef H5_HAVE_PARALLEL
    
    if (!H5CX_is_def_dxpl()) {
        H5CX_set_mpio_actual_chunk_opt(H5D_MPIO_NO_CHUNK_OPTIMIZATION);
        H5CX_set_mpio_actual_io_mode(H5D_MPIO_NO_COLLECTIVE);
    } 
#endif

    
    if (H5D__ioinfo_init(count, H5D_IO_OP_WRITE, dset_info, &io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize I/O info");

    
    if (count > 1)
        if (NULL == (store = (H5D_storage_t *)H5MM_malloc(count * sizeof(H5D_storage_t))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "couldn't allocate dset storage info array buffer");

    
    for (i = 0; i < count; i++) {
        bool    should_alloc_space = false; 
        haddr_t prev_tag           = HADDR_UNDEF;

        
        if (NULL == dset_info[i].dset)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset");
        if (NULL == dset_info[i].dset->oloc.file)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

        
        H5AC_tag(dset_info[i].dset->oloc.addr, &prev_tag);

        
        if (!dset_info[i].dset->shared->checked_filters) {
            
            if (H5D_flush_layout_to_dcpl(dset_info[i].dset) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to flush layout");

            if (H5Z_can_apply(dset_info[i].dset->shared->dcpl_id, dset_info[i].dset->shared->type_id) < 0)
                HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "can't apply filters");

            dset_info[i].dset->shared->checked_filters = true;
        } 

        
        if (0 == (H5F_INTENT(dset_info[i].dset->oloc.file) & H5F_ACC_RDWR))
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "no write intent on file");

        
        if (H5D__typeinfo_init(&io_info, &(dset_info[i]), dset_info[i].mem_type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info");

            
#ifdef H5_HAVE_PARALLEL
        if (H5F_HAS_FEATURE(dset_info[i].dset->oloc.file, H5FD_FEAT_HAS_MPI)) {
            
            
            
            if (H5T_is_vl_storage(dset_info[i].type_info.mem_type) > 0)
                HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                            "Parallel IO does not support writing VL or region reference datatypes yet");
        } 
        else {
            H5FD_mpio_xfer_t io_xfer_mode; 

            
            if (H5CX_get_io_xfer_mode(&io_xfer_mode) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI-I/O transfer mode");

            
            if (io_xfer_mode == H5FD_MPIO_COLLECTIVE)
                HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                            "collective access for MPI-based driver only");
        } 
#endif    

        
        dset_info[i].nelmts = H5S_GET_SELECT_NPOINTS(dset_info[i].mem_space);
        if (dset_info[i].nelmts != H5S_GET_SELECT_NPOINTS(dset_info[i].file_space))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "src and dest dataspaces have different number of elements selected");

        
        if (NULL == dset_info[i].buf.cvp) {
            
            if (dset_info[i].nelmts > 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no input buffer");

            
            dset_info[i].buf.cvp = &fake_char;
        } 

        
        if (!(H5S_has_extent(dset_info[i].file_space)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file dataspace does not have extent set");
        if (!(H5S_has_extent(dset_info[i].mem_space)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "memory dataspace does not have extent set");

        
        if (dset_info[i].nelmts > 0 &&
            true == H5S_SELECT_SHAPE_SAME(dset_info[i].mem_space, dset_info[i].file_space) &&
            H5S_GET_EXTENT_NDIMS(dset_info[i].mem_space) != H5S_GET_EXTENT_NDIMS(dset_info[i].file_space)) {
            ptrdiff_t buf_adj = 0;

            
            if (!orig_mem_space) {
                if (count > 1) {
                    
                    if (NULL == (orig_mem_space = (H5S_t **)H5MM_calloc(count * sizeof(H5S_t *))))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL,
                                    "couldn't allocate original memory space array buffer");
                }
                else
                    
                    orig_mem_space = &orig_mem_space_local;
            }

            
            orig_mem_space[i]      = dset_info[i].mem_space;
            dset_info[i].mem_space = NULL;

            
            if (H5S_select_construct_projection(orig_mem_space[i], &dset_info[i].mem_space,
                                                (unsigned)H5S_GET_EXTENT_NDIMS(dset_info[i].file_space),
                                                dset_info[i].type_info.src_type_size, &buf_adj) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to construct projected memory dataspace");
            assert(dset_info[i].mem_space);

            
            dset_info[i].buf.cvp = (const void *)(((const uint8_t *)dset_info[i].buf.cvp) + buf_adj);
        } 

        
        

        
        if (H5D__dset_ioinfo_init(dset_info[i].dset, &(dset_info[i]), &(store[i])) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up I/O operation");

        
        if (dset_info[i].dset->shared->dcpl_cache.pline.nused > 0)
            io_info.filtered_count++;

        
        should_alloc_space = dset_info[i].dset->shared->dcpl_cache.efl.nused == 0 &&
                             !(*dset_info[i].dset->shared->layout.ops->is_space_alloc)(
                                 &dset_info[i].dset->shared->layout.storage);

        
        if (!H5F_HAS_FEATURE(dset_info[i].dset->oloc.file, H5FD_FEAT_HAS_MPI))
            should_alloc_space = should_alloc_space && (dset_info[i].nelmts > 0);

        if (should_alloc_space) {
            hssize_t file_nelmts;    
            bool     full_overwrite; 

            
            if ((file_nelmts = H5S_GET_EXTENT_NPOINTS(dset_info[i].file_space)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                            "can't retrieve number of elements in file dataset");

            
            if (H5T_detect_class(dset_info[i].dset->shared->type, H5T_VLEN, false))
                full_overwrite = false;
            else
                full_overwrite = (bool)((hsize_t)file_nelmts == dset_info[i].nelmts ? true : false);

            
            if (H5D__alloc_storage(dset_info[i].dset, H5D_ALLOC_WRITE, full_overwrite, NULL) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize storage");
        } 

        
        
        if (dset_info[i].layout_ops.io_init &&
            (*dset_info[i].layout_ops.io_init)(&io_info, &(dset_info[i])) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize I/O info");
        dset_info[i].skip_io = false;
        io_op_init++;

        
        H5AC_tag(prev_tag, NULL);
    } 

    assert(io_op_init == count);

    
    if (H5D__typeinfo_init_phase2(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info (second phase)");

#ifdef H5_HAVE_PARALLEL
    
    if (H5D__ioinfo_adjust(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "unable to adjust I/O info for parallel or selection I/O");
#endif 

    
    if (H5D__typeinfo_init_phase3(&io_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up type info (third phase)");

    H5CX_set_no_selection_io_cause(io_info.no_selection_io_cause);

    
    if (io_info.md_io_ops.multi_write_md) {
        
        if (io_info.piece_count > 0) {
            assert(!io_info.sel_pieces);
            assert(io_info.pieces_added == 0);

            
            if (NULL ==
                (io_info.sel_pieces = H5MM_malloc(io_info.piece_count * sizeof(io_info.sel_pieces[0]))))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "unable to allocate array of selected pieces");
        }

        
        for (i = 0; i < count; i++)
            if (dset_info[i].layout_ops.mdio_init) {
                haddr_t prev_tag = HADDR_UNDEF;

                
                H5AC_tag(dset_info[i].dset->oloc.addr, &prev_tag);

                
                if ((dset_info[i].layout_ops.mdio_init)(&io_info, &(dset_info[i])) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't populate array of selected pieces");

                
                H5AC_tag(prev_tag, NULL);
            }

        
        if ((*io_info.md_io_ops.multi_write_md)(&io_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "can't write data");
    } 
    else {
        haddr_t prev_tag = HADDR_UNDEF;

        
        if (!H5D_LAYOUT_CB_PERFORM_IO(&io_info) && io_info.piece_count > 0) {
            if (NULL == (io_info.mem_spaces = H5MM_malloc(io_info.piece_count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for memory space list");
            if (NULL == (io_info.file_spaces = H5MM_malloc(io_info.piece_count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for file space list");
            if (NULL == (io_info.addrs = H5MM_malloc(io_info.piece_count * sizeof(haddr_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for piece address list");
            if (NULL == (io_info.element_sizes = H5MM_malloc(io_info.piece_count * sizeof(size_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for element size list");
            if (NULL == (io_info.wbufs = H5MM_malloc(io_info.piece_count * sizeof(const void *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                            "memory allocation failed for write buffer list");
            if (io_info.max_tconv_type_size > 0)
                if (NULL ==
                    (io_info.sel_pieces = H5MM_malloc(io_info.piece_count * sizeof(io_info.sel_pieces[0]))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "unable to allocate array of selected pieces");
        }

        
        for (i = 0; i < count; i++) {
            assert(!dset_info[i].skip_io);

            
            H5AC_tag(dset_info->dset->oloc.addr, &prev_tag);

            
            if ((*dset_info[i].io_ops.multi_write)(&io_info, &dset_info[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "can't write data");

            
            H5AC_tag(prev_tag, NULL);
        }

        
        if (!H5D_LAYOUT_CB_PERFORM_IO(&io_info)) {
            
            if (io_info.max_tconv_type_size > 0) {
                
                if (H5D__scatgath_write_select(&io_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "type conversion selection write failed");
            }
            else {
                
                H5_CHECK_OVERFLOW(io_info.pieces_added, size_t, uint32_t);
                if (H5F_shared_select_write(io_info.f_sh, H5FD_MEM_DRAW, (uint32_t)io_info.pieces_added,
                                            io_info.mem_spaces, io_info.file_spaces, io_info.addrs,
                                            io_info.element_sizes, io_info.wbufs) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "selection write failed");
            }
        }

#ifdef H5_HAVE_PARALLEL
        
        if (io_info.using_mpi_vfd) {
            H5FD_mpio_xfer_t xfer_mode; 

            
            if (H5CX_get_io_xfer_mode(&xfer_mode) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI-I/O transfer mode");

            
            if (xfer_mode == H5FD_MPIO_COLLECTIVE) {
                H5CX_set_mpio_actual_io_mode(io_info.actual_io_mode);

                
                if (io_info.use_select_io == H5D_SELECTION_IO_MODE_ON)
                    H5CX_set_mpio_actual_chunk_opt(H5D_MPIO_LINK_CHUNK);
            }
        }
#endif 
    }

done:
    
    for (i = 0; i < io_op_init; i++) {
        assert(!dset_info[i].skip_io);
        if (dset_info[i].layout_ops.io_term &&
            (*dset_info[i].layout_ops.io_term)(&io_info, &(dset_info[i])) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "unable to shut down I/O op info");
    }

    
    if (H5D__typeinfo_term(&io_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "unable to shut down type info");

    
    if (orig_mem_space) {
        for (i = 0; i < count; i++)
            if (orig_mem_space[i]) {
                if (H5S_close(dset_info[i].mem_space) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL,
                                "unable to shut down projected memory dataspace");
                dset_info[i].mem_space = orig_mem_space[i];
            }

        
        if (orig_mem_space != &orig_mem_space_local)
            H5MM_free(orig_mem_space);
    }

    
    H5MM_xfree(io_info.sel_pieces);

    
    H5MM_xfree(io_info.mem_spaces);
    H5MM_xfree(io_info.file_spaces);
    H5MM_xfree(io_info.addrs);
    H5MM_xfree(io_info.element_sizes);
    H5MM_xfree(io_info.wbufs);

    
    if (store != &store_local)
        H5MM_free(store);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__ioinfo_init(size_t count, H5D_io_op_type_t op_type, H5D_dset_io_info_t *dset_info,
                 H5D_io_info_t *io_info)
{
    H5D_selection_io_mode_t selection_io_mode;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(count > 0);
    assert(dset_info);
    assert(dset_info[0].dset->oloc.file);
    assert(io_info);

    
    memset(io_info, 0, sizeof(*io_info));

    
    io_info->op_type = op_type;
    io_info->f_sh    = count > 0 ? H5F_SHARED(dset_info[0].dset->oloc.file) : NULL;
    io_info->count   = count;

    

    
    io_info->dsets_info = dset_info;

    
    H5CX_get_selection_io_mode(&selection_io_mode);
    io_info->use_select_io = selection_io_mode;

    
    if (selection_io_mode == H5D_SELECTION_IO_MODE_OFF)
        io_info->no_selection_io_cause = H5D_SEL_IO_DISABLE_BY_API;

#ifdef H5_HAVE_PARALLEL

    
    if (count > 0)
        io_info->using_mpi_vfd = H5F_HAS_FEATURE(dset_info[0].dset->oloc.file, H5FD_FEAT_HAS_MPI);
#endif 

    
    if (op_type == H5D_IO_OP_READ)
        
        io_info->may_use_in_place_tconv = true;
    else
        
        H5CX_get_modify_write_buf(&io_info->may_use_in_place_tconv);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__dset_ioinfo_init(H5D_t *dset, H5D_dset_io_info_t *dset_info, H5D_storage_t *store)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dset);
    assert(dset->oloc.file);
    assert(dset_info->type_info.tpath);

    
    dset_info->dset  = dset;
    dset_info->store = store;

    
    dset_info->layout_ops = *dset->shared->layout.ops;

    
    dset_info->io_ops.multi_read  = dset->shared->layout.ops->ser_read;
    dset_info->io_ops.multi_write = dset->shared->layout.ops->ser_write;

    
    if (dset_info->type_info.is_xform_noop && dset_info->type_info.is_conv_noop) {
        
        dset_info->io_ops.single_read  = H5D__select_read;
        dset_info->io_ops.single_write = H5D__select_write;
    } 
    else {
        
        dset_info->io_ops.single_read  = H5D__scatgath_read;
        dset_info->io_ops.single_write = H5D__scatgath_write;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__typeinfo_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info, const H5T_t *mem_type)
{
    H5D_type_info_t  *type_info;
    const H5D_t      *dset;
    H5Z_data_xform_t *data_transform;      
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(mem_type);

    
    type_info = &dset_info->type_info;
    dset      = dset_info->dset;
    assert(dset);

    
    if (H5T_patch_vlen_file(dset->shared->type, H5F_VOL_OBJ(dset->oloc.file)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "can't patch VL datatype file pointer");

    
    memset(type_info, 0, sizeof(*type_info));

    
    type_info->mem_type  = mem_type;
    type_info->dset_type = dset->shared->type;

    if (io_info->op_type == H5D_IO_OP_WRITE) {
        type_info->src_type = mem_type;
        type_info->dst_type = dset->shared->type;
    } 
    else {
        type_info->src_type = dset->shared->type;
        type_info->dst_type = mem_type;
    } 

    
    if (NULL == (type_info->tpath = H5T_path_find(type_info->src_type, type_info->dst_type)))
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unable to convert between src and dest datatype");

    
    if (H5CX_get_data_transform(&data_transform) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data transform info");

    
    type_info->src_type_size = H5T_get_size(type_info->src_type);
    type_info->dst_type_size = H5T_get_size(type_info->dst_type);
    type_info->is_conv_noop  = H5T_path_noop(type_info->tpath);
    type_info->is_xform_noop = H5Z_xform_noop(data_transform);
    if (type_info->is_xform_noop && type_info->is_conv_noop) {
        type_info->cmpd_subset = NULL;
        type_info->need_bkg    = H5T_BKG_NO;
    } 
    else {
        H5T_bkg_t bkgr_buf_type; 

        
        if (H5CX_get_bkgr_buf_type(&bkgr_buf_type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve background buffer type");

        
        type_info->cmpd_subset = H5T_path_compound_subset(type_info->tpath);

        
        io_info->max_tconv_type_size =
            MAX3(io_info->max_tconv_type_size, type_info->src_type_size, type_info->dst_type_size);

        
        if ((io_info->op_type == H5D_IO_OP_WRITE) && H5T_detect_class(dset->shared->type, H5T_VLEN, false))
            type_info->need_bkg = H5T_BKG_YES;
        else {
            H5T_bkg_t path_bkg; 

            if ((path_bkg = H5T_path_bkg(type_info->tpath))) {
                
                type_info->need_bkg = bkgr_buf_type;
                type_info->need_bkg = MAX(path_bkg, type_info->need_bkg);
            } 
            else
                type_info->need_bkg = H5T_BKG_NO; 
        }                                         
    }                                             

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__typeinfo_init_phase2(H5D_io_info_t *io_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);

    
    if (io_info->use_select_io == H5D_SELECTION_IO_MODE_DEFAULT) {
        if (H5F_has_vector_select_io(io_info->dsets_info[0].dset->oloc.file,
                                     io_info->op_type == H5D_IO_OP_WRITE))
            io_info->use_select_io = H5D_SELECTION_IO_MODE_ON;
        else {
            io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
            io_info->no_selection_io_cause |= H5D_SEL_IO_DEFAULT_OFF;
        }
    }

    
    if (io_info->max_tconv_type_size && io_info->use_select_io != H5D_SELECTION_IO_MODE_OFF) {
        size_t max_temp_buf; 
        size_t i;            

        

        
        for (i = 0; i < io_info->count; i++) {
            H5D_type_info_t *type_info = &io_info->dsets_info[i].type_info;

            
            if (type_info->need_bkg) {
                hsize_t bkg_buf_hsize;

                
                H5_CHECKED_ASSIGN(bkg_buf_hsize, hsize_t, io_info->bkg_buf_size, size_t);
                bkg_buf_hsize += io_info->dsets_info[i].nelmts * type_info->dst_type_size;
                io_info->bkg_buf_size = (size_t)bkg_buf_hsize;
                if (H5_UNLIKELY((hsize_t)io_info->bkg_buf_size != bkg_buf_hsize))
                    io_info->tconv_buf_overflow = true;

                
                if (type_info->need_bkg == H5T_BKG_YES)
                    io_info->must_fill_bkg = true;
            }
        }

        
        if (H5CX_get_max_temp_buf(&max_temp_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve max. temp. buf size");

        
        if (io_info->tconv_buf_size > max_temp_buf || io_info->tconv_buf_overflow) {
            io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
            io_info->no_selection_io_cause |= H5D_SEL_IO_TCONV_BUF_TOO_SMALL;
            io_info->tconv_buf_size = 0;
            io_info->bkg_buf_size   = 0;
            io_info->must_fill_bkg  = false;
        }
        if (io_info->bkg_buf_size > max_temp_buf || io_info->tconv_buf_overflow) {
            io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
            io_info->no_selection_io_cause |= H5D_SEL_IO_BKG_BUF_TOO_SMALL;
            io_info->tconv_buf_size = 0;
            io_info->bkg_buf_size   = 0;
            io_info->must_fill_bkg  = false;
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_PARALLEL

static herr_t
H5D__ioinfo_adjust(H5D_io_info_t *io_info)
{
    H5D_t *dset0;               
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);

    
    assert(io_info->dsets_info[0].dset);
    dset0 = io_info->dsets_info[0].dset;
    assert(dset0->oloc.file);

    
    if (io_info->using_mpi_vfd) {
        H5FD_mpio_xfer_t xfer_mode; 
        htri_t           opt;       

        
        if (H5CX_get_io_xfer_mode(&xfer_mode) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI-I/O transfer mode");

        
        if (MPI_COMM_NULL == (io_info->comm = H5F_mpi_get_comm(dset0->oloc.file)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't retrieve MPI communicator");

        
        if ((opt = H5D__mpio_opt_possible(io_info)) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_BADRANGE, FAIL, "invalid check for direct IO dataspace ");

        
        if (opt == true) {
            
            
            if (io_info->use_select_io == H5D_SELECTION_IO_MODE_OFF) {
                io_info->md_io_ops.multi_read_md   = H5D__collective_read;
                io_info->md_io_ops.multi_write_md  = H5D__collective_write;
                io_info->md_io_ops.single_read_md  = H5D__mpio_select_read;
                io_info->md_io_ops.single_write_md = H5D__mpio_select_write;
            } 
        }     
        else {
            
            if (io_info->op_type == H5D_IO_OP_WRITE) {
                bool mpi_file_sync_required = false;
                if (H5F_shared_get_mpi_file_sync_required(io_info->f_sh, &mpi_file_sync_required) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get MPI file_sync_required flag");

                if (mpi_file_sync_required)
                    HGOTO_ERROR(
                        H5E_DATASET, H5E_NO_INDEPENDENT, FAIL,
                        "Can't perform independent write when MPI_File_sync is required by ROMIO driver.");
            }

            
            if (io_info->op_type == H5D_IO_OP_WRITE) {
                size_t i;

                
                for (i = 0; i < io_info->count; i++)
                    if (io_info->dsets_info[i].dset->shared->dcpl_cache.pline.nused > 0)
                        break;

                
                if (i < io_info->count) {
                    int comm_size = 0;

                    
                    if ((comm_size = H5F_shared_mpi_get_size(io_info->f_sh)) < 0)
                        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get MPI communicator size");

                    if (comm_size > 1) {
                        char local_no_coll_cause_string[512];
                        char global_no_coll_cause_string[512];

                        if (H5D__mpio_get_no_coll_cause_strings(local_no_coll_cause_string, 512,
                                                                global_no_coll_cause_string, 512) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                        "can't get reasons for breaking collective I/O");

                        HGOTO_ERROR(H5E_IO, H5E_NO_INDEPENDENT, FAIL,
                                    "Can't perform independent write with filters in pipeline.\n"
                                    "    The following caused a break from collective I/O:\n"
                                    "        Local causes: %s\n"
                                    "        Global causes: %s",
                                    local_no_coll_cause_string, global_no_coll_cause_string);
                    }
                }
            }

            
            if (xfer_mode == H5FD_MPIO_COLLECTIVE) {
                
                if (H5CX_set_io_xfer_mode(H5FD_MPIO_INDEPENDENT) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set MPI-I/O transfer mode");
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

static herr_t
H5D__typeinfo_init_phase3(H5D_io_info_t *io_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);

    
    if (io_info->max_tconv_type_size) {
        void *tconv_buf; 
        void *bkgr_buf;  

        
        if (H5CX_get_tconv_buf(&tconv_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve temp. conversion buffer pointer");
        if (H5CX_get_bkgr_buf(&bkgr_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "can't retrieve background conversion buffer pointer");

        
        if (io_info->use_select_io == H5D_SELECTION_IO_MODE_ON) {
            

            
            
            if (io_info->tconv_buf_size > 0) {
                if (NULL == (io_info->tconv_buf = H5FL_BLK_MALLOC(type_conv, io_info->tconv_buf_size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                "memory allocation failed for type conversion");
                io_info->tconv_buf_allocated = true;
            }

            
            if (io_info->bkg_buf_size > 0) {
                if (NULL == (io_info->bkg_buf = H5FL_BLK_MALLOC(type_conv, io_info->bkg_buf_size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                "memory allocation failed for type conversion");
                io_info->bkg_buf_allocated = true;
            }
        }
        else {
            
            size_t max_temp_buf; 
            size_t target_size;  
            size_t i;

            
            assert(io_info->use_select_io == H5D_SELECTION_IO_MODE_OFF);

            
            if (H5CX_get_max_temp_buf(&max_temp_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve max. temp. buf size");

            
            target_size = max_temp_buf;

            
            if (target_size < io_info->max_tconv_type_size) {
                bool default_buffer_info; 

                
                default_buffer_info =
                    (bool)((H5D_TEMP_BUF_SIZE == max_temp_buf) && (NULL == tconv_buf) && (NULL == bkgr_buf));

                
                if (default_buffer_info)
                    
                    target_size = io_info->max_tconv_type_size;
                else
                    
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "temporary buffer max size is too small");
            } 

            
            if (NULL == (io_info->tconv_buf = (uint8_t *)tconv_buf)) {
                
                if (NULL == (io_info->tconv_buf = H5FL_BLK_MALLOC(type_conv, target_size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                "memory allocation failed for type conversion");
                io_info->tconv_buf_allocated = true;
            } 

            
            for (i = 0; i < io_info->count; i++) {
                H5D_type_info_t *type_info = &io_info->dsets_info[i].type_info;

                
                type_info->request_nelmts =
                    target_size / MAX(type_info->src_type_size, type_info->dst_type_size);

                
                if (type_info->need_bkg && (NULL == io_info->bkg_buf) &&
                    (NULL == (io_info->bkg_buf = (uint8_t *)bkgr_buf))) {
                    
                    if (NULL == (io_info->bkg_buf = H5FL_BLK_MALLOC(type_conv, target_size)))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                    "memory allocation failed for background conversion");
                    io_info->bkg_buf_allocated = true;
                }
            }
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__typeinfo_term(H5D_io_info_t *io_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    if (io_info->tconv_buf_allocated) {
        assert(io_info->tconv_buf);
        (void)H5FL_BLK_FREE(type_conv, io_info->tconv_buf);
    } 
    if (io_info->bkg_buf_allocated) {
        assert(io_info->bkg_buf);
        (void)H5FL_BLK_FREE(type_conv, io_info->bkg_buf);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
