#' Cut data for completers analysis
#'
#' Subsets the data to all subjects randomized by the specified date,
#' and prepares the data for analysis. This is a wrapper for [cut_data_by_date()]
#' typically used with a date determined by [cut_date_for_completers()].
#'
#' @param data Data generated by [nb_sim()].
#' @param cut_date Calendar time (relative to trial start) at which to cut the data.
#' @param event_gap Gap duration after each event during which no new events are counted.
#'   Can be a numeric value (default `0`) or a function returning a numeric value.
#'   The time at risk is reduced by the sum of these gaps (truncated by the cut date).
#'
#' @return A data frame with one row per subject randomized prior to `cut_date`.
#'   Contains the truncated follow-up time (`tte`) and total number of observed events (`events`).
#'
#' @export
#'
#' @examples
#' enroll_rate <- data.frame(rate = 20 / (5 / 12), duration = 5 / 12)
#' fail_rate <- data.frame(treatment = c("Control", "Experimental"), rate = c(0.5, 0.3))
#' dropout_rate <- data.frame(
#'   treatment = c("Control", "Experimental"),
#'   rate = c(0.1, 0.05), duration = c(100, 100)
#' )
#' sim <- nb_sim(enroll_rate, fail_rate, dropout_rate, max_followup = 2, n = 20)
#' # Find date when 5 subjects have completed
#' date_5 <- cut_date_for_completers(sim, 5)
#' # Get analysis dataset for this cut date (includes partial follow-up)
#' cut_completers(sim, date_5)
cut_completers <- function(data, cut_date, event_gap = 0) {
  cut_data_by_date(data, cut_date, event_gap = event_gap)
}

#' Find calendar date for target completer count
#'
#' Finds the calendar time (since start of randomization) at which a specified
#' number of subjects have completed their follow-up.
#'
#' @param data A data frame of simulated data, typically from [nb_sim()] or
#'   [nb_sim_seasonal()].
#' @param target_completers Integer. The target number of completers.
#'
#' @return Numeric. The calendar date when `target_completers` is achieved.
#'   If the dataset contains fewer than `target_completers` completers, returns the maximum
#'   calendar time in the dataset and prints a message.
#'
#' @export
#'
#' @examples
#' enroll_rate <- data.frame(rate = 20 / (5 / 12), duration = 5 / 12)
#' fail_rate <- data.frame(treatment = c("Control", "Experimental"), rate = c(0.5, 0.3))
#' dropout_rate <- data.frame(
#'   treatment = c("Control", "Experimental"),
#'   rate = c(0.1, 0.05), duration = c(100, 100)
#' )
#' sim <- nb_sim(enroll_rate, fail_rate, dropout_rate, max_followup = 2, n = 20)
#' cut_date_for_completers(sim, target_completers = 5)
cut_date_for_completers <- function(data, target_completers) {
  dt <- data.table::as.data.table(data)

  followup_var <- NULL
  if ("tte" %in% names(dt)) {
    followup_var <- "tte"
  } else if ("end" %in% names(dt)) {
    # nb_sim_seasonal() stores follow-up interval end times in `end`
    followup_var <- "end"
  } else {
    stop("Data must contain a follow-up time column named 'tte' (nb_sim) or 'end' (nb_sim_seasonal).")
  }

  if (!"id" %in% names(dt)) {
    stop("Data must contain an 'id' column.")
  }

  # Ensure we have a numeric completion time for each record.
  # Prefer calendar columns when available; otherwise compute enroll_time + follow-up.
  if ("calendar_time" %in% names(dt)) {
    dt[, completion_time := calendar_time]
  } else if ("cal_end" %in% names(dt)) {
    # Convert Date to relative time (years) using earliest calendar start as origin.
    origin_date <- min(dt$cal_start, na.rm = TRUE)
    dt[, completion_time := as.numeric(cal_end - origin_date) / 365.25]
  } else {
    if (!"enroll_time" %in% names(dt)) {
      stop("Data must contain 'enroll_time' to compute completion times.")
    }
    dt[, completion_time := enroll_time + get(followup_var)]
  }

  # Determine max_followup (from the observed maximum follow-up in the dataset)
  max_f <- max(dt[[followup_var]], na.rm = TRUE)

  # One row per subject: take their max follow-up and max completion time
  last_rows <- dt[, .(
    followup = max(get(followup_var), na.rm = TRUE),
    completion_time = max(completion_time, na.rm = TRUE)
  ), by = id]

  # Completers are those who reached (approximately) the max follow-up
  completers <- last_rows[followup >= max_f - 1e-8]

  if (nrow(completers) < target_completers) {
    message(sprintf("Only %d completers in trial (target: %d)", nrow(completers), target_completers))
    return(max(dt$completion_time, na.rm = TRUE))
  }

  # Sort completion times
  completion_times <- sort(completers$completion_time)

  # Return the date the target-th completer finished
  return(completion_times[target_completers])
}
