% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/performance.R
\name{performance}
\alias{performance}
\title{Performance Measures for Precision Matrix Estimation}
\usage{
performance(hatOmega, Omega)
}
\arguments{
\item{hatOmega}{A numeric \eqn{d \times d} matrix giving the estimated
precision matrix.}

\item{Omega}{A numeric \eqn{d \times d} matrix giving the reference
(typically true) precision matrix.}
}
\value{
A data frame of S3 class \code{"performance"}, with one row per performance
metric and two columns:
\describe{
\item{measure}{The name of each performance metric. The reported metrics
include: sparsity, Frobenius norm loss, Kullback-Leibler divergence,
quadratic norm loss, spectral norm loss, true positive, true negative,
false positive, false negative, true positive rate, false positive rate,
F1 score, and Matthews correlation coefficient.}
\item{value}{The corresponding numeric value.}
}
}
\description{
Compute a collection of loss-based and structure-based measures to evaluate
the performance of an estimated precision matrix.
}
\details{
Let \eqn{\Omega_{d \times d}} and \eqn{\hat{\Omega}_{d \times d}} be
the reference (true) and estimated precision matrices, respectively, with
\eqn{\Sigma = \Omega^{-1}} being the corresponding covariance matrix.
Edges are defined by nonzero off-diagonal entries in the upper triangle of
the precision matrices.

Sparsity is treated as a structural summary, while the remaining measures
are grouped into loss-based measures, raw confusion-matrix counts, and
classification-based (structure-recovery) measures.

"sparsity": Sparsity is computed as the proportion of zero entries among
the off-diagonal elements in the upper triangle of \eqn{\hat{\Omega}}.

\strong{Loss-based measures:}
\itemize{
\item "Frobenius": Frobenius (Hilbert-Schmidt) norm loss
\eqn{= \Vert \Omega - \hat{\Omega} \Vert_F}.
\item "KL": Kullback-Leibler divergence
\eqn{= \mathrm{tr}(\Sigma \hat{\Omega}) - \log\det(\Sigma \hat{\Omega}) - d}.
\item "quadratic": Quadratic norm loss
\eqn{= \Vert \Sigma \hat{\Omega} - I_d \Vert_F^2}.
\item "spectral": Spectral (operator) norm loss
\eqn{= \Vert \Omega - \hat{\Omega} \Vert_{2,2} = e_1},
where \eqn{e_1^2} is the largest eigenvalue of \eqn{(\Omega - \hat{\Omega})^2}.
}

\strong{Confusion-matrix counts:}
\itemize{
\item "TP": True positive \eqn{=} number of edges
in both \eqn{\Omega} and \eqn{\hat{\Omega}}.
\item "TN": True negative \eqn{=} number of edges
in neither \eqn{\Omega} nor \eqn{\hat{\Omega}}.
\item "FP": False positive \eqn{=} number of edges
in \eqn{\hat{\Omega}} but not in \eqn{\Omega}.
\item "FN": False negative \eqn{=} number of edges
in \eqn{\Omega} but not in \eqn{\hat{\Omega}}.
}

\strong{Classification-based (structure-recovery) measures:}
\itemize{
\item "TPR": True positive rate (TPR), recall, sensitivity
\eqn{= \mathrm{TP} / (\mathrm{TP} + \mathrm{FN})}.
\item "FPR": False positive rate (FPR)
\eqn{= \mathrm{FP} / (\mathrm{FP} + \mathrm{TN})}.
\item "F1": \eqn{F_1} score
\eqn{= 2\,\mathrm{TP} / (2\,\mathrm{TP} + \mathrm{FN} + \mathrm{FP})}
\item "MCC": Matthews correlation coefficient (MCC)
\eqn{= (\mathrm{TP}\times\mathrm{TN} - \mathrm{FP}\times\mathrm{FN}) /
\sqrt{(\mathrm{TP}+\mathrm{FP})(\mathrm{TP}+\mathrm{FN})
(\mathrm{TN}+\mathrm{FP})(\mathrm{TN}+\mathrm{FN})}}
}

The following table summarizes the confusion matrix and related rates:
\tabular{lllll}{
\tab \strong{Predicted Positive} \tab \strong{Predicted Negative} \tab \tab \cr
\strong{Real Positive} (P) \tab True positive (TP) \tab False negative (FN)
\tab True positive rate (TPR), recall, sensitivity = TP / P = 1 - FNR
\tab False negative rate (FNR) = FN / P = 1 - TPR \cr
\strong{Real Negative} (N) \tab False positive (FP) \tab True negative (TN)
\tab False positive rate (FPR) = FP / N = 1 - TNR
\tab True negative rate (TNR), specificity = TN / N = 1 - FPR \cr
\tab Positive predictive value (PPV), precision = TP / (TP + FP) = 1 - FDR
\tab False omission rate (FOR) = FN / (TN + FN) = 1 - NPV \tab \tab \cr
\tab False discovery rate (FDR) = FP / (TP + FP) = 1 - PPV
\tab Negative predictive value (NPV) = TN / (TN + FN) = 1 - FOR \tab \tab \cr
}
}
\examples{
library(grasps)

## reproducibility for everything
set.seed(1234)

## block-structured precision matrix based on SBM
sim <- gen_prec_sbm(d = 30, K = 3,
                    within.prob = 0.25, between.prob = 0.05,
                    weight.dists = list("gamma", "unif"),
                    weight.paras = list(c(shape = 20, rate = 10),
                                        c(min = 0, max = 5)),
                    cond.target = 100)
## visualization
plot(sim)

## n-by-d data matrix
library(MASS)
X <- mvrnorm(n = 20, mu = rep(0, 30), Sigma = sim$Sigma)

## adapt, BIC
res <- grasps(X = X, membership = sim$membership, penalty = "adapt", crit = "BIC")

## visualization
plot(res)

## performance
performance(hatOmega = res$hatOmega, Omega = sim$Omega)
}
