% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tm.R
\name{tm_fwd}
\alias{tm_fwd}
\alias{tm_rev}
\alias{tm_exact_fwd}
\alias{tm_exact_rev}
\title{Transverse Mercator projection}
\usage{
tm_fwd(x, lon0, k0 = 0.9996)

tm_rev(x, y, lon0, k0 = 0.9996)

tm_exact_fwd(x, lon0, k0 = 0.9996)

tm_exact_rev(x, y, lon0, k0 = 0.9996)
}
\arguments{
\item{x}{For forward conversion: a two-column matrix or data frame of
coordinates (longitude, latitude) in decimal degrees.
For reverse conversion: numeric vector of x (easting) coordinates in meters.}

\item{lon0}{Central meridian in decimal degrees. Can be a vector to specify
different central meridians for each point.}

\item{k0}{Scale factor on the central meridian. Default is 0.9996 (UTM).
Common values: 0.9996 (UTM), 1.0 (many national grids), 0.9999 (some state planes).}

\item{y}{Numeric vector of y (northing) coordinates in meters (reverse only).}
}
\value{
Data frame with columns:
\itemize{
\item For forward conversion:
\itemize{
\item \code{x}: Easting in meters
\item \code{y}: Northing in meters
\item \code{convergence}: Grid convergence in degrees
\item \code{scale}: Scale factor at the point
\item \code{lon}, \code{lat}: Input coordinates (echoed)
\item \code{lon0}: Central meridian (echoed)
}
\item For reverse conversion:
\itemize{
\item \code{lon}: Longitude in decimal degrees
\item \code{lat}: Latitude in decimal degrees
\item \code{convergence}: Grid convergence in degrees
\item \code{scale}: Scale factor at the point
\item \code{x}, \code{y}: Input coordinates (echoed)
\item \code{lon0}: Central meridian (echoed)
}
}
}
\description{
Convert geographic coordinates to/from Transverse Mercator projection
with user-specified central meridian and scale factor.

Two versions are provided:
\itemize{
\item \code{tm_fwd()}/\code{tm_rev()}: Series approximation, fast, accurate to ~5 nanometers
\item \code{tm_exact_fwd()}/\code{tm_exact_rev()}: Exact formulation, slower but accurate everywhere
}
}
\details{
The Transverse Mercator projection is a conformal cylindrical projection
commonly used for:
\itemize{
\item UTM (Universal Transverse Mercator) zones
\item Many national and state coordinate systems
\item Large-scale topographic mapping
}

Unlike \code{utmups_fwd()} which automatically selects UTM zones, these functions
allow you to specify any central meridian and scale factor.

The series approximation (\code{tm_fwd}/\code{tm_rev}) is accurate to ~5 nanometers
within 3900 km of the central meridian. The exact version
(\code{tm_exact_fwd}/\code{tm_exact_rev}) is slower but works accurately everywhere.

The \code{lon0} parameter is vectorized, allowing different central meridians
for each point (useful for processing data across multiple zones).
}
\examples{
# Basic Transverse Mercator (like UTM zone 55)
pts <- cbind(lon = c(147, 148, 149), lat = c(-42, -43, -44))
tm_fwd(pts, lon0 = 147, k0 = 0.9996)

# Compare with UTM
utmups_fwd(pts)

# Custom scale factor (k0 = 1.0)
tm_fwd(pts, lon0 = 147, k0 = 1.0)

# Different central meridian for each point
tm_fwd(pts, lon0 = c(147, 148, 149), k0 = 0.9996)

# Round-trip conversion
fwd <- tm_fwd(pts, lon0 = 147, k0 = 0.9996)
tm_rev(fwd$x, fwd$y, lon0 = 147, k0 = 0.9996)

# Exact version for high precision or extreme locations
tm_exact_fwd(pts, lon0 = 147, k0 = 0.9996)
}
\seealso{
\code{\link[=utmups_fwd]{utmups_fwd()}} for automatic UTM zone selection.
}
