% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate_edgelist.R
\name{generate_edgelist}
\alias{generate_edgelist}
\title{Export null modelling results}
\usage{
generate_edgelist(
  nullnet,
  signif.level = 0.95,
  export.null = FALSE,
  edge.cols = c("#67A9CF", "#F7F7F7", "#EF8A62")
)
}
\arguments{
\item{nullnet}{An object of class nullnet from \code{generate_null_net}}

\item{signif.level}{An optional value specifying the threshold used for
testing for 'significant' deviations from the null model.  Defaults to 0.95}

\item{export.null}{An optional logical value specifying whether export should
be limited to those interactions that were present in the observed network
or should include any additional interactions present across iterations
of the null network.  Depending upon the data and any forbidden links
specified, additional interactions may be present in the modelled networks.
Defaults to FALSE (only observed interactions are exported).}

\item{edge.cols}{An optional character vector of length three specifying
potential colours for network links when used with a suitable plotting
function: in sequence, these should represent i) interactions that are
weaker than expected, ii) consistent with the null model and iii) stronger
than expected. The default is a colourblind friendly blue, white and red
scheme, using Colorbrewer's Red-Blue colour scheme (Brewer 2017).}
}
\value{
A data frame where each row represents the interaction observed
  between a pair of consumer and resource species, and with the following
  column headings:
  \describe{
    \item{\code{Consumer}}{Name of the consumer species (node)}
    \item{\code{Resource}}{Name of the resource species (node)}
    \item{\code{Observed}}{Strength of the observed interaction}
    \item{\code{Null}}{Mean strength of the interaction across the iterations
      of the null model}
    \item{\code{SES}}{The standardised effect size for the interaction}
    \item{\code{Test}}{Whether the observed interaction is significantly
      \strong{stronger} than expected under the null model, \strong{weaker}
      or consistent with the null model \strong{ns}}
  }
}
\description{
Exports the observed network alongside the mean interaction strengths
  calculated from the null model and the significance test results in a
  standard format that can be imported into other network analysis packages
  e.g. \code{igraph} (Csardi & Nepusz 2006) or \code{cheddar} (Hudson
  \emph{et al}. 2013).  This provides access to a wide range of plotting and
  analysis tools, especially for non-bipartite networks.
}
\examples{
set.seed(1234)
bs.null <- generate_null_net(Broadstone, Broadstone.prey,
                             data.type = "counts", sims = 10,
                             r.weights = Broadstone.fl)
BS.export <- generate_edgelist(bs.null, signif.level = 0.95,
                               edge.cols = c("#67A9CF", "black", "#EF8A62"))

if (requireNamespace("igraph", quietly = TRUE)) {
 net.1 <- igraph::graph_from_edgelist(as.matrix(BS.export[, c("Resource",
                                                              "Consumer")]),
                                      directed = TRUE)

 # Add in the null modelling results
 igraph::E(net.1)$obs.str <- BS.export$Observed
 igraph::E(net.1)$test.res <- BS.export$Test
 igraph::E(net.1)$edge.cols <- BS.export$edge.col

 igraph::plot.igraph(net.1, layout = igraph::layout_in_circle,
                     edge.color = igraph::E(net.1)$edge.cols,
                     edge.width = sqrt(igraph::E(net.1)$obs.str),
                     edge.arrow.size = .4)
}

}
\references{
Brewer, C.A. (2017) \url{https://colorbrewer2.org/}

  Csardi, G. & Nepusz, T. (2006) The igraph software package for complex
  network research. \emph{InterJournal, Complex Systems}, \strong{1695}.

  Vaughan, I.P., Gotelli, N.J., Memmott, J., Pearson, C.E., Woodward, G. &
  Symondson, W.O.C. (2018) econullnetr: an R package using null models to
  analyse the structure of ecological networks and identify resource
  selection. \emph{Methods in Ecology and Evolution}, \strong{9}, 728--733.
}
\seealso{
\code{\link{generate_null_net}}
}
