% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Design-methods.R
\name{simulate,DADesign-method}
\alias{simulate,DADesign-method}
\title{Simulate outcomes from a time-to-DLT augmented CRM design}
\usage{
\S4method{simulate}{DADesign}(
  object,
  nsim = 1L,
  seed = NULL,
  truthTox,
  truthSurv,
  trueTmax = NULL,
  args = NULL,
  firstSeparate = FALSE,
  deescalate = TRUE,
  mcmcOptions = McmcOptions(),
  DA = TRUE,
  parallel = FALSE,
  nCores = min(parallel::detectCores(), 5),
  derive = list(),
  ...
)
}
\arguments{
\item{object}{the \code{\link{DADesign}} object we want to simulate data from}

\item{nsim}{(\code{count})\cr the number of simulations (default: 1)}

\item{seed}{see \code{\link[=set_seed]{set_seed()}}}

\item{truthTox}{(\code{function})\cr a function which takes as input a dose (vector) and returns the
true probability (vector) for toxicity and the time DLT occurs. Additional
arguments can be supplied in \code{args}.}

\item{truthSurv}{(\code{function})\cr a CDF which takes as input a time (vector) and returns
the true cumulative probability (vector) that the DLT would occur conditioning on the patient
has DLTs.}

\item{trueTmax}{(\code{number} or \code{NULL})\cr the true maximum time at which DLTs can occur.
Note that this must be larger than \code{Tmax} from the \code{object}'s base data, which is
the length of the DLT window, i.e. until which time DLTs are officially declared
as such and used in the trial.}

\item{args}{(\code{data.frame})\cr data frame with arguments for the \code{truthTox} function. The
column names correspond to the argument names, the rows to the values of the
arguments. The rows are appropriately recycled in the \code{nsim}
simulations. In order to produce outcomes from the posterior predictive
distribution, e.g, pass an \code{object} that contains the data observed so
far, \code{truthTox} contains the \code{prob} function from the model in
\code{object}, and \code{args} contains posterior samples from the model.}

\item{firstSeparate}{(\code{flag})\cr enroll the first patient separately from the rest of
the cohort? (not default) If yes, the cohort will be closed if a DLT occurs
in this patient.}

\item{deescalate}{(\code{flag})\cr allow deescalation when a DLT occurs in cohorts with lower dose
level? (default: TRUE)}

\item{mcmcOptions}{(\link{McmcOptions})\cr object of class \code{\link{McmcOptions}},
giving the MCMC options for each evaluation in the trial. By default,
the standard options are used.}

\item{DA}{(\code{flag})\cr use dose-adaptation rules? (default: TRUE)}

\item{parallel}{(\code{flag})\cr should the simulation runs be parallelized across the
clusters of the computer? (not default)}

\item{nCores}{(\code{count})\cr how many cores should be used for parallel computing?
Defaults to the number of cores on the machine, maximum 5.}

\item{derive}{(\code{list})\cr a named list of functions which derives statistics, based on the
vector of posterior MTD samples. Each list element must therefore accept
one and only one argument, which is a numeric vector, and return a number.}

\item{...}{not used}
}
\value{
an object of class \code{\link{Simulations}}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

This method simulates dose escalation trials using time-to-DLT data,
where the timing of dose-limiting toxicities is explicitly modeled.
}
\examples{
# nolint start

# Define the dose-grid and PEM parameters
emptydata <- DataDA(
  doseGrid = c(0.1, 0.5, 1, 1.5, 3, 6, seq(from = 10, to = 80, by = 2)),
  Tmax = 60
)

# Initialize the mDA-CRM model
npiece_ <- 10
Tmax_ <- 60

lambda_prior <- function(k) {
  npiece_ / (Tmax_ * (npiece_ - k + 0.5))
}

model <- DALogisticLogNormal(
  mean = c(-0.85, 1),
  cov = matrix(c(1, -0.5, -0.5, 1), nrow = 2),
  ref_dose = 56,
  npiece = npiece_,
  l = as.numeric(t(apply(as.matrix(c(1:npiece_), 1, npiece_), 2, lambda_prior))),
  c_par = 2
)

# Choose the rule for dose increments
myIncrements <- IncrementsRelative(
  intervals = c(0, 20),
  increments = c(1, 0.33)
)

myNextBest <- NextBestNCRM(
  target = c(0.2, 0.35),
  overdose = c(0.35, 1),
  max_overdose_prob = 0.25
)

# Choose the rule for the cohort-size
mySize1 <- CohortSizeRange(
  intervals = c(0, 30),
  cohort_size = c(1, 3)
)
mySize2 <- CohortSizeDLT(
  intervals = c(0, 1),
  cohort_size = c(1, 3)
)
mySize <- maxSize(mySize1, mySize2)

# Choose the rule for stopping
myStopping1 <- StoppingTargetProb(
  target = c(0.2, 0.35),
  prob = 0.5
)
myStopping2 <- StoppingMinPatients(nPatients = 50)

myStopping <- (myStopping1 | myStopping2) | StoppingMissingDose()

# Choose the safety window
mysafetywindow <- SafetyWindowConst(c(6, 2), 7, 7)

# Initialize the design
design <- DADesign(
  model = model,
  increments = myIncrements,
  nextBest = myNextBest,
  stopping = myStopping,
  cohort_size = mySize,
  data = emptydata,
  safetyWindow = mysafetywindow,
  startingDose = 3
)

## set up truth curves
myTruth <- probFunction(model, alpha0 = 2, alpha1 = 3)
curve(myTruth(x), from = 0, to = 100, ylim = c(0, 1))

exp_cond.cdf <- function(x, onset = 15) {
  a <- pexp(28, 1 / onset, lower.tail = FALSE)
  1 - (pexp(x, 1 / onset, lower.tail = FALSE) - a) / (1 - a)
}

# set up simulation settings
options <- McmcOptions(
  burnin = 10,
  step = 1,
  samples = 200
)

\donttest{
mySims <- simulate(design,
  args = NULL,
  truthTox = myTruth,
  truthSurv = exp_cond.cdf,
  trueTmax = 80,
  nsim = 2,
  seed = 819,
  mcmcOptions = options,
  firstSeparate = TRUE,
  deescalate = FALSE,
  parallel = FALSE
)
}

# nolint end
}
