% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mock_bids.R
\name{create_mock_bids}
\alias{create_mock_bids}
\title{Create a Mock BIDS Project Object}
\usage{
create_mock_bids(
  project_name,
  participants,
  file_structure,
  dataset_description = NULL,
  event_data = list(),
  confound_data = list(),
  create_stub = FALSE,
  stub_path = NULL,
  prep_dir = "derivatives/fmriprep"
)
}
\arguments{
\item{project_name}{A character string for the project name.}

\item{participants}{Either a \code{data.frame} mirroring \code{participants.tsv} content
(must include 'participant_id') or a character vector of participant IDs
(e.g., \code{c("01", "02")}). If IDs are given, a minimal \code{part_df} is created.}

\item{file_structure}{A \code{data.frame} or \code{tibble} defining the files in the
mock structure. Each row represents a file. Required columns:
\code{subid}, \code{datatype}, \code{suffix}. Optional BIDS entity columns: \code{session},
\code{task}, \code{run}, \code{acq}, \code{rec}, \code{dir}, \code{space}, \code{desc}, etc. Must also include
a logical column \code{fmriprep} indicating if the file belongs in the derivatives
directory specified by \code{prep_dir}.}

\item{dataset_description}{A list representing the \code{dataset_description.json}
content. Defaults to a minimal valid description.}

\item{event_data}{A named list where names are the \emph{relative paths} of
\code{events.tsv} files (e.g., "sub-01/func/sub-01_task-A_run-1_events.tsv")
and values are the corresponding \code{tibble} or \code{data.frame} content for
those files. These paths must correspond to files defined in \code{file_structure}
with a \code{suffix} like "events.tsv".}

\item{confound_data}{A named list where names are relative paths of
confound TSV files within the derivatives directory and values are
their \code{tibble} or \code{data.frame} content. Paths must match files defined
in \code{file_structure}.}

\item{create_stub}{Logical (default \code{FALSE}). If \code{TRUE}, write a stub BIDS
directory structure to disk at \code{stub_path}. Zero-byte files are created
except for \code{participants.tsv}, \code{dataset_description.json}, and \code{events.tsv}
files specified in \code{event_data}.}

\item{stub_path}{Character string, the path where the stub directory will be
created. Required if \code{create_stub = TRUE}.}

\item{prep_dir}{Character string, the path relative to the root for derivatives
(default "derivatives/fmriprep"). This path structure will be used both in
the internal \code{data.tree} and for stub creation.}
}
\value{
An object of class \code{mock_bids_project}.
}
\description{
Generates an in-memory representation of a BIDS project, suitable for
testing and demonstration without requiring actual data files. Can optionally
create a "stub" directory structure on disk.
}
\examples{
\donttest{
# --- Example Setup ---
participants_df <- tibble::tibble(participant_id = c("01", "02"), age = c(25, 30))

file_structure_df <- tibble::tribble(
  ~subid, ~session, ~datatype, ~task,   ~run, ~suffix,                  ~fmriprep, ~desc,
  "01",   NA,       "anat",    NA,      NA,   "T1w.nii.gz",             FALSE,     NA,
  "01",   NA,       "func",    "taskA", "01", "bold.nii.gz",            FALSE,     NA,
  "01",   NA,       "func",    "taskA", "01", "events.tsv",             FALSE,     NA,
  "02",   "test",   "anat",    NA,      NA,   "T1w.nii.gz",             FALSE,     NA,
  "02",   "test",   "func",    "taskA", "01", "bold.nii.gz",            FALSE,     NA,
  "02",   "test",   "func",    "taskA", "01", "events.tsv",             FALSE,     NA,
  # Example derivative
  "01",   NA,       "func",    "taskA", "01", "preproc_bold.nii.gz",    TRUE,      "preproc"
)

# Define event data (paths must match generated structure)
event_data_list <- list()
event_data_list[["sub-01/func/sub-01_task-taskA_run-01_events.tsv"]] <- tibble::tibble(
  onset = c(1.0, 5.0), duration = c(0.5, 0.5), trial_type = c("condA", "condB")
)
event_data_list[["sub-02/ses-test/func/sub-02_ses-test_task-taskA_run-01_events.tsv"]] <-
  tibble::tibble(
    onset = c(1.5, 5.5), duration = c(0.5, 0.5), trial_type = c("condC", "condD")
)

# Create the mock project (in memory only)
mock_proj <- create_mock_bids(
  project_name = "MockTaskA",
  participants = participants_df,
  file_structure = file_structure_df,
  event_data = event_data_list
)

# Create the mock project and write stubs
mock_proj_stub <- create_mock_bids(
  project_name = "MockTaskA_stub",
  participants = c("01", "02"), # Example using just IDs
  file_structure = file_structure_df,
  event_data = event_data_list,
  create_stub = TRUE,
  stub_path = tempdir() # Use a temporary directory for example
)

# --- Using the Mock Project ---
print(mock_proj)
print(participants(mock_proj))
print(tasks(mock_proj))
print(sessions(mock_proj)) # Should return "test"

print(func_scans(mock_proj, subid = "01"))
print(event_files(mock_proj, subid = "02", session = "test"))

# Read the injected event data
events_sub1 <- read_events(mock_proj, subid = "01")
print(events_sub1)
if (nrow(events_sub1) > 0) print(tidyr::unnest(events_sub1, cols = data))

# Search for derivatives
print(search_files(mock_proj, suffix = "preproc_bold.nii.gz"))

# Check stub directory (if created)
stub_files <- list.files(mock_proj_stub$path, recursive = TRUE)
print(head(stub_files))

# Read one injected stub event file if present
stub_event_path <- file.path(mock_proj_stub$path, names(event_data_list)[1])
if (file.exists(stub_event_path)) {
  print(readLines(stub_event_path, n = 1))
}

# Cleanup is intentionally omitted in this example.
}
}
