% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hdi.R
\name{hdi}
\alias{hdi}
\alias{hdi.numeric}
\alias{hdi.data.frame}
\alias{hdi.brmsfit}
\alias{hdi.get_predicted}
\title{Highest Density Interval (HDI)}
\usage{
hdi(x, ...)

\method{hdi}{numeric}(x, ci = 0.95, verbose = TRUE, ...)

\method{hdi}{data.frame}(x, ci = 0.95, rvar_col = NULL, verbose = TRUE, ...)

\method{hdi}{brmsfit}(
  x,
  ci = 0.95,
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  verbose = TRUE,
  ...
)

\method{hdi}{get_predicted}(x, ci = 0.95, use_iterations = FALSE, verbose = TRUE, ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution, or a data frame of such
vectors. Can also be a Bayesian model. \strong{bayestestR} supports a wide range
of models (see, for example, \code{methods("hdi")}) and not all of those are
documented in the 'Usage' section, because methods for other classes mostly
resemble the arguments of the \code{.numeric} or \code{.data.frame}methods.}

\item{...}{Currently not used.}

\item{ci}{Value or vector of probability of the (credible) interval - CI
(between 0 and 1) to be estimated. Default to \code{.95} (\verb{95\%}).}

\item{verbose}{Toggle off warnings.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should variables for fixed effects (\code{"fixed"}), random effects
(\code{"random"}) or both (\code{"all"}) be returned? Only applies to mixed models. May
be abbreviated.

For models of from packages \strong{brms} or \strong{rstanarm} there are additional
options:
\itemize{
\item \code{"fixed"} returns fixed effects.
\item \code{"random_variance"} return random effects parameters (variance and
correlation components, e.g. those parameters that start with \code{sd_} or
\code{cor_}).
\item \code{"grouplevel"} returns random effects group level estimates, i.e. those
parameters that start with \code{r_}.
\item \code{"random"} returns both \code{"random_variance"} and \code{"grouplevel"}.
\item \code{"all"} returns fixed effects and random effects variances.
\item \code{"full"} returns all parameters.
}}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{use_iterations}{Logical, if \code{TRUE} and \code{x} is a \code{get_predicted} object,
(returned by \code{\link[insight:get_predicted]{insight::get_predicted()}}), the function is applied to the
iterations instead of the predictions. This only applies to models that return
iterations for predicted values (e.g., \code{brmsfit} models).}
}
\value{
A data frame with following columns:
\itemize{
\item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a
vector, this column is missing.
\item \code{CI} The probability of the credible interval.
\item \code{CI_low}, \code{CI_high} The lower and upper credible interval limits for the parameters.
}
}
\description{
Compute the \strong{Highest Density Interval (HDI)} of posterior distributions.
All points within this interval have a higher probability density than points
outside the interval. The HDI can be used in the context of uncertainty
characterisation of posterior distributions as \strong{Credible Interval (CI)}.
}
\details{
Unlike equal-tailed intervals (see \code{\link[=eti]{eti()}}) that typically exclude
\verb{2.5\%} from each tail of the distribution and always include the median, the
HDI is \emph{not} equal-tailed and therefore always includes the mode(s) of posterior
distributions. While this can be useful to better represent the credibility
mass of a distribution, the HDI also has some limitations. See \code{\link[=spi]{spi()}} for
details.

The \href{https://easystats.github.io/bayestestR/articles/credible_interval.html}{\verb{95\%} or \verb{89\%} Credible Intervals (CI)}
are two reasonable ranges to characterize the uncertainty related to the
estimation (see \href{https://easystats.github.io/bayestestR/articles/credible_interval.html}{here}
for a discussion about the differences between these two values).

The \verb{89\%} intervals (\code{ci = 0.89}) are deemed to be more stable than, for
instance, \verb{95\%} intervals (\emph{Kruschke, 2014}). An effective sample size
of at least 10.000 is recommended if one wants to estimate \verb{95\%} intervals
with high precision (\emph{Kruschke, 2014, p. 183ff}). Unfortunately, the
default number of posterior samples for most Bayes packages (e.g., \code{rstanarm}
or \code{brms}) is only 4.000 (thus, you might want to increase it when fitting
your model). Moreover, 89 indicates the arbitrariness of interval limits -
its only remarkable property is being the highest prime number that does not
exceed the already unstable \verb{95\%} threshold (\emph{McElreath, 2015}).

However, \verb{95\%} has some \href{https://easystats.github.io/blog/posts/bayestestr_95/}{advantages too}.
For instance, it shares (in the case of a normal posterior distribution) an
intuitive relationship with the standard deviation and it conveys a more
accurate image of the (artificial) bounds of the distribution. Also, because
it is wider, it makes analyses more conservative (i.e., the probability of
covering zero is larger for the \verb{95\%} CI than for lower ranges such as \verb{89\%}),
which is a good thing in the context of the reproducibility crisis.

A \verb{95\%} equal-tailed interval (ETI) has \verb{2.5\%} of the distribution on either
side of its limits. It indicates the 2.5th percentile and the 97.5th
percentile. In symmetric distributions, the two methods of computing credible
intervals, the ETI and the \link[=hdi]{HDI}, return similar results.

This is not the case for skewed distributions. Indeed, it is possible that
parameter values in the ETI have lower credibility (are less probable) than
parameter values outside the ETI. This property seems undesirable as a summary
of the credible values in a distribution.

On the other hand, the ETI range does change when transformations are applied
to the distribution (for instance, for a log odds scale to probabilities):
the lower and higher bounds of the transformed distribution will correspond
to the transformed lower and higher bounds of the original distribution.
On the contrary, applying transformations to the distribution will change
the resulting HDI.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (all(insight::check_if_installed(c("rstanarm", "brms", "emmeans", "BayesFactor"), quietly = TRUE))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

posterior <- rnorm(1000)
hdi(posterior, ci = 0.89)
hdi(posterior, ci = c(0.80, 0.90, 0.95))

hdi(iris[1:4])
hdi(iris[1:4], ci = c(0.80, 0.90, 0.95))
\donttest{
model <- suppressWarnings(
  rstanarm::stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
)
hdi(model)
hdi(model, ci = c(0.80, 0.90, 0.95))

hdi(emmeans::emtrends(model, ~1, "wt", data = mtcars))

model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
hdi(model)
hdi(model, ci = c(0.80, 0.90, 0.95))

bf <- BayesFactor::ttestBF(x = rnorm(100, 1, 1))
hdi(bf)
hdi(bf, ci = c(0.80, 0.90, 0.95))
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Kruschke, J. (2014). Doing Bayesian data analysis: A tutorial with R, JAGS,
and Stan. Academic Press.
\item McElreath, R. (2015). Statistical rethinking: A Bayesian course with
examples in R and Stan. Chapman and Hall/CRC.
}
}
\seealso{
Other interval functions, such as \code{\link[=hdi]{hdi()}}, \code{\link[=eti]{eti()}}, \code{\link[=bci]{bci()}},
\code{\link[=spi]{spi()}}, \code{\link[=si]{si()}}.

Other ci: 
\code{\link{bci}()},
\code{\link{ci}()},
\code{\link{eti}()},
\code{\link{si}()},
\code{\link{spi}()}
}
\author{
Credits go to \strong{ggdistribute} and \href{https://github.com/mikemeredith/HDInterval}{\strong{HDInterval}}.
}
\concept{ci}
