% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcsd.R
\name{bcsd}
\alias{bcsd}
\title{Bias Correction Spatial Disaggregation (BCSD) for statistical downscaling}
\usage{
bcsd(
  coarse_data,
  fine_data,
  method = "bilinear",
  n_quantiles = 100,
  reference_period = NULL,
  extrapolate = TRUE,
  normalize = TRUE
)
}
\arguments{
\item{coarse_data}{A 3D array of coarse resolution input data. The two first dimensions are the spatial coordinates
(e.g., latitude and longitude) in grid, and the third dimension refers to the training samples (e.g. time).}

\item{fine_data}{A 3D array of fine resolution output data. The two first dimensions are the spatial coordinates
(e.g., latitude and longitude) in grid, and the third dimension refers to the training samples (e.g. time).}

\item{method}{Character. Interpolation method that is used by \link[raster]{resample} to perform predictions.
The options are ('bilinear', 'ngb') that refer to bilinear and nearest neighbor interpolation, respectively. Default: 'bilinear'.}

\item{n_quantiles}{Integer. Number of quantiles for bias correction. Default: 100.}

\item{reference_period}{Vector. Start and end indices for the reference period. Default: NULL (use all data).}

\item{extrapolate}{Logical. Indicating whether to extrapolate corrections outside the range of the training data. Default: TRUE.}

\item{normalize}{Logical. Indicating whether to normalize data before processing. Default: TRUE.}
}
\value{
Object of class \code{BCSD} containing the trained model components:
\item{quantile_map}{Quantile mapping function for bias correction.}
\item{interpolation_params}{Parameters for spatial interpolation.}
\item{axis_names}{Names of the axes in the fine data.}
\item{scalers}{List of scalers. If \code{normalize = TRUE},
the list contains scalers \code{coarse} and \code{fine}
for the coarse and fine data, respectively.
If \code{normalize = FALSE}, the list is empty.}
\item{model_params}{List of all model parameters.}
}
\description{
Implements the BCSD method for statistical downscaling of climate data.
The approach consists of two main steps: (1) bias correction using quantile mapping
and (2) spatial disaggregation using interpolation.
}
\details{
The BCSD method is a statistical downscaling technique that combines bias correction
and spatial disaggregation. It uses quantile mapping to correct biases in coarse resolution data
and then applies spatial interpolation to disaggregate the data to a finer resolution.

The function allows for the interpolation methods "bilinear" and "ngb", that perform bilinear and nearest neighbor interpolation, respectively. For more information on these methods, please refer to the documentation for \link[raster]{resample}.
The function provides the option to normalize data before processing, by using the \code{normalize} argument.
The quantile mapping step involves calculating quantiles from the coarse data and mapping them to the fine data.
The interpolation step uses the specified method to create a fine resolution grid from the coarse data.
}
\examples{
# Simple example with random data
coarse <- array(rnorm(8 * 8 * 10), dim = c(8, 8, 10))  # e.g. lat x lon x time 
fine <- array(rnorm(16 * 16 * 10), dim = c(16, 16, 10))    # e.g. lat x lon x time
model <- bcsd(coarse, fine, method = "bilinear", n_quantiles = 100)
coarse_new <- array(rnorm(8 * 8 * 3), dim = c(8, 8, 3))  # e.g. lat x lon x time 
predictions <- predict(model, coarse_new)

}
