#' Visualization Functions for RegimeChange
#'
#' @description
#' Functions for visualizing changepoint detection results, including
#' data with changepoints, posterior distributions, and segment comparisons.
#'
#' @name visualization
#' @noRd
NULL

#' Plot Regime Change Detection Results
#'
#' @description
#' Create visualizations of changepoint detection results.
#'
#' @param x A regime_result object
#' @param type Type of plot:
#'   \itemize{
#'     \item \code{"data"}: Data with changepoints marked (default)
#'     \item \code{"segments"}: Data colored by segment
#'     \item \code{"posterior"}: Posterior probability of change (Bayesian methods)
#'     \item \code{"diagnostic"}: Diagnostic statistics
#'     \item \code{"runlength"}: Run length distribution (BOCPD)
#'   }
#' @param show_ci Show confidence intervals if available
#' @param show_segments Color segments differently
#' @param title Plot title
#' @param ... Additional arguments passed to ggplot
#'
#' @return A ggplot2 object
#'
#' @examples
#' data <- c(rnorm(100), rnorm(100, mean = 2))
#' result <- detect_regimes(data)
#' plot(result)
#' plot(result, type = "segments")
#'
#' @export
plot.regime_result <- function(x,
                               type = c("data", "segments", "posterior",
                                        "diagnostic", "runlength"),
                               show_ci = TRUE,
                               show_segments = TRUE,
                               title = NULL,
                               ...) {
  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    cli::cli_abort("Package 'ggplot2' required for plotting")
  }
  
  type <- match.arg(type)
  
  p <- switch(type,
              data = plot_data_with_changepoints(x, show_ci, show_segments, title, ...),
              segments = plot_segments(x, title, ...),
              posterior = plot_posterior(x, title, ...),
              diagnostic = plot_diagnostic(x, title, ...),
              runlength = plot_runlength(x, title, ...)
  )
  
  p
}

#' @noRd
plot_data_with_changepoints <- function(result, show_ci = TRUE,
                                        show_segments = TRUE, title = NULL, ...) {
  data <- result$data
  changepoints <- result$changepoints
  
  if (is.matrix(data)) {
    data <- data[, 1]
    if (is.null(title)) title <- "Changepoint Detection (Variable 1)"
  }
  
  n <- length(data)
  
  df <- data.frame(
    time = seq_len(n),
    value = data
  )
  
  if (show_segments) {
    df$segment <- as.factor(create_segment_labels(changepoints, n))
  }
  
  if (show_segments && length(changepoints) > 0) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = value, color = segment)) +
      ggplot2::geom_line(linewidth = 0.5) +
      ggplot2::geom_point(size = 0.8, alpha = 0.6)
  } else {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = value)) +
      ggplot2::geom_line(color = "steelblue", linewidth = 0.5) +
      ggplot2::geom_point(color = "steelblue", size = 0.8, alpha = 0.6)
  }
  
  if (length(changepoints) > 0) {
    p <- p + ggplot2::geom_vline(
      xintercept = changepoints,
      linetype = "dashed",
      color = "red",
      linewidth = 0.8
    )
    
    if (show_ci && !is.null(result$confidence_intervals)) {
      for (ci in result$confidence_intervals) {
        if (!is.na(ci$lower) && !is.na(ci$upper)) {
          p <- p + ggplot2::annotate(
            "rect",
            xmin = ci$lower, xmax = ci$upper,
            ymin = -Inf, ymax = Inf,
            alpha = 0.1, fill = "red"
          )
        }
      }
    }
  }
  
  if (is.null(title)) {
    title <- sprintf("Changepoint Detection (%s)", result$method)
  }
  
  p <- p +
    ggplot2::labs(
      title = title,
      subtitle = sprintf("%d changepoint(s) detected", result$n_changepoints),
      x = "Time",
      y = "Value"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(
      legend.position = if (show_segments) "bottom" else "none",
      plot.title = ggplot2::element_text(hjust = 0.5),
      plot.subtitle = ggplot2::element_text(hjust = 0.5)
    )
  
  if (show_segments) {
    p <- p + ggplot2::scale_color_brewer(palette = "Set1", name = "Segment")
  }
  
  p
}

#' @noRd
plot_segments <- function(result, title = NULL, ...) {
  data <- result$data
  changepoints <- result$changepoints
  segments <- result$segments
  
  if (is.matrix(data)) {
    data <- data[, 1]
  }
  
  n <- length(data)
  
  df <- data.frame(
    time = seq_len(n),
    value = data,
    segment = as.factor(create_segment_labels(changepoints, n))
  )
  
  seg_means <- sapply(segments, function(s) s$params$mean)
  seg_df <- data.frame(
    segment = as.factor(seq_along(segments)),
    start = sapply(segments, `[[`, "start"),
    end = sapply(segments, `[[`, "end"),
    mean = seg_means
  )
  
  p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = value)) +
    ggplot2::geom_point(ggplot2::aes(color = segment), size = 1, alpha = 0.6) +
    ggplot2::geom_segment(
      data = seg_df,
      ggplot2::aes(x = start, xend = end, y = mean, yend = mean, color = segment),
      linewidth = 1.5
    ) +
    ggplot2::labs(
      title = if (is.null(title)) "Detected Segments" else title,
      x = "Time",
      y = "Value"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::scale_color_brewer(palette = "Set1", name = "Segment")
  
  p
}

#' @noRd
plot_posterior <- function(result, title = NULL, ...) {
  if (is.null(result$prob_change)) {
    cli::cli_warn("No posterior probability available for this method")
    return(plot_data_with_changepoints(result, title = title, ...))
  }
  
  prob_change <- result$prob_change
  changepoints <- result$changepoints
  threshold <- result$threshold
  
  n <- length(prob_change)
  
  df <- data.frame(
    time = seq_len(n),
    prob = prob_change
  )
  
  p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = prob)) +
    ggplot2::geom_area(fill = "steelblue", alpha = 0.3) +
    ggplot2::geom_line(color = "steelblue", linewidth = 0.8)
  
  if (!is.null(threshold)) {
    p <- p + ggplot2::geom_hline(
      yintercept = threshold,
      linetype = "dashed",
      color = "red"
    )
  }
  
  if (length(changepoints) > 0) {
    cp_df <- data.frame(
      time = changepoints,
      prob = prob_change[changepoints]
    )
    p <- p + ggplot2::geom_point(
      data = cp_df,
      color = "red",
      size = 3
    )
  }
  
  p <- p +
    ggplot2::labs(
      title = if (is.null(title)) "Posterior Probability of Changepoint" else title,
      subtitle = sprintf("Method: %s | Threshold: %.2f",
                         result$method,
                         if (!is.null(threshold)) threshold else NA),
      x = "Time",
      y = "P(Changepoint)"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::ylim(0, 1)
  
  p
}

#' @noRd
plot_diagnostic <- function(result, title = NULL, ...) {
  if (!is.null(result$statistic)) {
    stat <- result$statistic
    if (!is.null(result$R)) stat <- result$R
    
    df <- data.frame(
      time = seq_along(stat),
      statistic = stat
    )
    
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = statistic)) +
      ggplot2::geom_line(color = "steelblue", linewidth = 0.8)
    
    if (!is.null(result$threshold)) {
      p <- p + ggplot2::geom_hline(
        yintercept = result$threshold,
        linetype = "dashed",
        color = "red"
      )
    }
    
    p <- p +
      ggplot2::labs(
        title = if (is.null(title)) "Detection Statistic" else title,
        x = "Time",
        y = "Statistic"
      ) +
      ggplot2::theme_minimal()
    
    return(p)
  }
  
  if (!is.null(result$prob_change)) {
    return(plot_posterior(result, title, ...))
  }
  
  plot_data_with_changepoints(result, title = title, ...)
}

#' @noRd
plot_runlength <- function(result, title = NULL, ...) {
  if (is.null(result$posterior) || is.null(result$map_run_length)) {
    cli::cli_warn("Run length distribution not available")
    return(plot_data_with_changepoints(result, title = title, ...))
  }
  
  map_rl <- result$map_run_length
  
  df <- data.frame(
    time = seq_along(map_rl),
    run_length = map_rl
  )
  
  p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = run_length)) +
    ggplot2::geom_line(color = "steelblue", linewidth = 0.8) +
    ggplot2::geom_point(size = 0.5, alpha = 0.5)
  
  if (length(result$changepoints) > 0) {
    p <- p + ggplot2::geom_vline(
      xintercept = result$changepoints,
      linetype = "dashed",
      color = "red"
    )
  }
  
  p <- p +
    ggplot2::labs(
      title = if (is.null(title)) "MAP Run Length Over Time" else title,
      subtitle = "Drops indicate detected changepoints",
      x = "Time",
      y = "Run Length (time since last change)"
    ) +
    ggplot2::theme_minimal()
  
  p
}

#' Create Multi-Panel Summary Plot
#'
#' @description
#' Creates a comprehensive multi-panel visualization of detection results.
#'
#' @param result A regime_result object
#' @param ... Additional arguments
#'
#' @return A combined ggplot object
#'
#' @export
plot_summary <- function(result, ...) {
  if (!requireNamespace("patchwork", quietly = TRUE)) {
    cli::cli_warn("Package 'patchwork' recommended for summary plots. Using single plot.")
    return(plot.regime_result(result, ...))
  }
  
  p1 <- plot.regime_result(result, type = "data", show_ci = TRUE)
  p2 <- plot.regime_result(result, type = "segments")
  
  if (!is.null(result$prob_change)) {
    p3 <- plot.regime_result(result, type = "posterior")
    combined <- patchwork::wrap_plots(p1, p2, p3, ncol = 1)
  } else {
    combined <- patchwork::wrap_plots(p1, p2, ncol = 1)
  }
  
  combined + patchwork::plot_annotation(
    title = sprintf("Changepoint Detection Summary (%s)", result$method)
  )
}

#' Create Interactive Plot
#'
#' @description
#' Creates an interactive version of the changepoint plot using plotly.
#'
#' @param result A regime_result object
#' @param ... Additional arguments
#'
#' @return A plotly object
#'
#' @export
plot_interactive <- function(result, ...) {
  if (!requireNamespace("plotly", quietly = TRUE)) {
    cli::cli_abort("Package 'plotly' required for interactive plots")
  }
  
  p <- plot.regime_result(result, type = "data", ...)
  
  plotly::ggplotly(p)
}

#' Plot Multiple Results for Comparison
#'
#' @description
#' Creates a faceted plot comparing results from different methods.
#'
#' @param ... Multiple regime_result objects
#' @param names Names for each result
#'
#' @return A ggplot object
#'
#' @export
plot_compare <- function(..., names = NULL) {
  results <- list(...)
  
  if (is.null(names)) {
    names <- sapply(results, function(r) r$method)
  }
  
  dfs <- lapply(seq_along(results), function(i) {
    r <- results[[i]]
    data <- if (is.matrix(r$data)) r$data[, 1] else r$data
    
    data.frame(
      time = seq_along(data),
      value = data,
      method = names[i],
      segment = as.factor(create_segment_labels(r$changepoints, length(data)))
    )
  })
  
  df <- do.call(rbind, dfs)
  
  cp_df <- do.call(rbind, lapply(seq_along(results), function(i) {
    r <- results[[i]]
    if (length(r$changepoints) > 0) {
      data.frame(
        method = names[i],
        changepoint = r$changepoints
      )
    } else {
      NULL
    }
  }))
  
  p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = value)) +
    ggplot2::geom_line(color = "steelblue", linewidth = 0.5) +
    ggplot2::facet_wrap(~method, ncol = 1, scales = "free_y")
  
  if (!is.null(cp_df) && nrow(cp_df) > 0) {
    p <- p + ggplot2::geom_vline(
      data = cp_df,
      ggplot2::aes(xintercept = changepoint),
      linetype = "dashed",
      color = "red"
    )
  }
  
  p <- p +
    ggplot2::labs(
      title = "Method Comparison",
      x = "Time",
      y = "Value"
    ) +
    ggplot2::theme_minimal()
  
  p
}