library(testthat)
library(RealSurvSim)


# Sample dataset creation
set.seed(123)
dat <- dats$Liang

# Rename columns to match the function requirements
names(dat) <- c("V1", "V2", "V3")

test_that("Test 1: Validates input types and values", {
  expect_error(RealSurvSim(123, "V1", "V2", "V3"), "dat must be a data frame.")
  expect_error(RealSurvSim(dat, "V1", "V2", "V3", reps = -1), "reps should be a single positive integer.")
  expect_error(RealSurvSim(dat, "V1", "V2", "V3", random_seed = "123"), "random_seed should be a single numeric value.")
  expect_error(RealSurvSim(dat, "V1", "V2", "V3", n = "50"), "n should be a vector of non-negative integers.")
  #expect_error(RealSurvSim(dat, "V1", "V2", "V3", distribs = c("invalid", "exp", "exp", "exp")), "distribs contains invalid distribution names.")
})

test_that("Test 2: Simulates data using 'case' bootstrap method with default n", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == nrow(dat)))
  expect_true(all(sapply(result$datasets, function(x) all(c("V1", "V2", "V3") %in% colnames(x)))))
})

test_that("Test 3: Simulates data using 'case' bootstrap method with arbitrary n", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, n = c(100, 50), simul_type = "case")
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == 150)) # Sum of 100 and 50
  expect_true(all(sapply(result$datasets, function(x) all(c("V1", "V2", "V3") %in% colnames(x)))))
})

test_that("Test 4: Simulates data using 'cond' bootstrap method", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "cond")
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == nrow(dat)))
  expect_true(all(sapply(result$datasets, function(x) all(c("V1", "V2", "V3") %in% colnames(x)))))
})

test_that("Test 5: Simulates data using 'distr' method", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp"))
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == nrow(dat)))
  expect_true(all(sapply(result$datasets, function(x) all(c("V1", "V2", "V3") %in% colnames(x)))))
})

test_that("Test 6: Simulates data using 'KDE' method", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE")
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == nrow(dat)))
  expect_true(all(sapply(result$datasets, function(x) all(c("V1", "V2", "V3") %in% colnames(x)))))
})

test_that("Test 7: Handles missing 'n' parameter correctly", {
  result <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
  expect_true(is.list(result))
  expect_equal(length(result$datasets), 10)
  expect_true(all(sapply(result$datasets, nrow) == nrow(dat)))
})
#
# test_that("Test 8: Handles empty dataframe", {
#   empty_dat <- dat[0, ]
#   expect_error(RealSurvSim(empty_dat, "V1", "V2", "V3", reps = 10, simul_type = "case"), "dat must be a data frame.")
#   expect_error(RealSurvSim(empty_dat, "V1", "V2", "V3", reps = 10, simul_type = "cond"), "dat must be a data frame.")
#   expect_error(RealSurvSim(empty_dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp")), "dat must be a data frame.")
#   expect_error(RealSurvSim(empty_dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE"), "dat must be a data frame.")
# })

test_that("Test 9: Reproducibility with set seed", {
  set.seed(123)
  result_1 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
  set.seed(123)
  result_2 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
  expect_identical(result_1, result_2)

  set.seed(123)
  result_3 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "cond")
  set.seed(123)
  result_4 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "cond")
  expect_identical(result_3, result_4)

  set.seed(123)
  result_5 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp"))
  set.seed(123)
  result_6 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp"))
  expect_identical(result_5, result_6)

  set.seed(123)
  result_7 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE")
  set.seed(123)
  result_8 <- RealSurvSim(dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE")
  expect_identical(result_7, result_8)
})
#
# test_that("Test 10: Handles dataframe with only events", {
#   only_events_dat <- dat %>% filter(V2 == 1)
#   result_case <- RealSurvSim(only_events_dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
#   result_cond <- RealSurvSim(only_events_dat, "V1", "V2", "V3", reps = 10, simul_type = "cond")
#   result_distr <- RealSurvSim(only_events_dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp"))
#   result_kde <- RealSurvSim(only_events_dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE")
#
#   expect_true(is.list(result_case))
#   expect_true(is.list(result_cond))
#   expect_true(is.list(result_distr))
#   expect_true(is.list(result_kde))
#
#   expect_equal(length(result_case$datasets), 10)
#   expect_equal(length(result_cond$datasets), 10)
#   expect_equal(length(result_distr$datasets), 10)
#   expect_equal(length(result_kde$datasets), 10)
#
#   expect_true(all(sapply(result_case$datasets, nrow) == nrow(only_events_dat)))
#   expect_true(all(sapply(result_cond$datasets, nrow) == nrow(only_events_dat)))
#   expect_true(all(sapply(result_distr$datasets, nrow) == nrow(only_events_dat)))
#   expect_true(all(sapply(result_kde$datasets, nrow) == nrow(only_events_dat)))
# })
#
# test_that("Test 11: Handles dataframe with only censored", {
#   only_censored_dat <- dat %>% filter(V2 == 0)
#   result_case <- RealSurvSim(only_censored_dat, "V1", "V2", "V3", reps = 10, simul_type = "case")
#   result_cond <- RealSurvSim(only_censored_dat, "V1", "V2", "V3", reps = 10, simul_type = "cond")
#   result_distr <- RealSurvSim(only_censored_dat, "V1", "V2", "V3", reps = 10, simul_type = "distr", distribs = c("exp", "exp", "exp", "exp"))
#   result_kde <- RealSurvSim(only_censored_dat, "V1", "V2", "V3", reps = 10, simul_type = "KDE")
#
#   expect_true(is.list(result_case))
#   expect_true(is.list(result_cond))
#   expect_true(is.list(result_distr))
#   expect_true(is.list(result_kde))
#
#   expect_equal(length(result_case$datasets), 10)
#   expect_equal(length(result_cond$datasets), 10)
#   expect_equal(length(result_distr$datasets), 10)
#   expect_equal(length(result_kde$datasets), 10)
#
#   expect_true(all(sapply(result_case$datasets, nrow) == nrow(only_censored_dat)))
#   expect_true(all(sapply(result_cond$datasets, nrow) == nrow(only_censored_dat)))
#   expect_true(all(sapply(result_distr$datasets, nrow) == nrow(only_censored_dat)))
#   expect_true(all(sapply(result_kde$datasets, nrow) == nrow(only_censored_dat)))
# })

# All tests completed
#cat("All tests for RealSurvSim function executed successfully with the liang_recon dataset.\n")
