#' @title Post-process MCMC Output for Profile GLMM
#'
#' @description This function performs essential post-processing of the MCMC output generated by
#' \code{profileGLMM_Gibbs}. It calculates posterior means and credible intervals for fixed effects
#' and, optionally, computes a representative cluster partition using Least Squares (LS) or
#' Ng's spectral clustering (NG). It also estimates cluster characteristics such as centroids,
#' probability vectors, and outcome effects for the chosen partition.
#'
#' @param MCMC_Obj An object of class \code{pglmm_mcmc} (the output of \code{profileGLMM_Gibbs}).
#' @param modeClus A character string specifying the clustering method.
#' Options are \code{'NG'} (Ng's spectral clustering, default) or \code{'LS'} (Least Squares clustering).
#' @param comp_cooc A logical value. If \code{TRUE} (default), the co-occurrence matrix is computed
#' and clustering is performed. If \code{FALSE}, only the population parameters are processed.
#' @param alpha A numeric value between 0 and 1, specifying the significance level for
#' credible intervals. Defaults to \code{0.05} (95\% CIs).
#'
#' @return An object of class \code{pglmm_fit}. This is a list containing:
#' \itemize{
#'   \item \code{coocMat}: The co-occurrence matrix of MCMC cluster assignments.
#'   \item \code{clust}: A list of representative clustering results (if \code{comp_cooc = TRUE}),
#'   including the optimal partition (\code{Zstar}), number of clusters (\code{Kstar}),
#'   and cluster-specific parameters (\code{cen}, \code{pvec}, \code{gamma}).
#'   \item \code{pop}: A list containing the posterior means and credible intervals for fixed effects.
#' }
#'
#' @export
#' @importFrom stats quantile
#' @importFrom Matrix Matrix
#' @importFrom utils head
#' @importFrom Spectrum estimate_k cluster_similarity
#'
#' @examples
#' # Load MCMC_Obj, the result of profileGLMM_Gibbs()
#' data("examp")
#' MCMC_Obj = examp$MCMC_Obj
#'
#' # Post-process the results
#' post_Obj = profileGLMM_postProcess(MCMC_Obj, modeClus='LS')
#'
#' # Removing the cooc matrix to save space
#' post_Obj$coocMat = NULL
#'
profileGLMM_postProcess = function(MCMC_Obj, modeClus='NG', comp_cooc = TRUE, alpha = 0.05){

  if (!inherits(MCMC_Obj, "pglmm_mcmc")) stop("Input must be a pglmm_mcmc object.")

  nSim = dim(MCMC_Obj$Z)[2]
  nUCat = dim(MCMC_Obj$pvec)[1]
  nUCont = dim(MCMC_Obj$muClus)[1]
  qLat = dim(MCMC_Obj$gamma)[1]


  # post processing of the population parameters
  pop = {}


  betaFE = apply(MCMC_Obj$beta, 1, mean)
  betaFECI = apply(MCMC_Obj$beta, 1, quantile,c(alpha/2,1-alpha/2))
  pop$betaFE = as.data.frame(rbind(betaFE,betaFECI))
  colnames(pop$betaFE) = MCMC_Obj$names$FE
  rownames(pop$betaFE)[1] = 'mean'



  # post processing of the profile clustering and outcome effect
  if(comp_cooc){
    cooc = create_co_occurrence_matrix_cpp(MCMC_Obj$Z)
    if (modeClus=='LS'){
      idx = find_ls_optimal_partition(cooc,
                                      MCMC_Obj$Z)
      Zstar = MCMC_Obj$Z[,idx]
      Kstar = length(unique(Zstar))
      mode = 'LS'
      clus = TRUE
    }else if (modeClus=='NG'){
      tmp = estimate_k(cooc, maxk = 15,showplots=FALSE)
      diffs <- diff(tmp$evals)
      diffs <- diffs[-1]
      Kstar <- which.max(abs(diffs[1:15 - 1])) + 1
      Zstar = cluster_similarity(cooc, k = Kstar, specalg = "Ng")

      mode = 'NG'
      clus = TRUE
    }else{
      warning('No valid clustering method provided returning only the coocurency
          matrix')
      rep_clust = NULL
      clus = FALSE
    }

    if(clus){
      # Compute the cluster caracteristics
      pvecPost = array(0,dim =c(nUCat,Kstar,nSim))
      centroids = array(0,dim =c(nUCont,Kstar,nSim))
      coVar = array(0,dim = c(nUCont,nUCont,Kstar))
      gamma = array(0,dim =c(qLat,Kstar,nSim))
      if (!is.null(Zstar)){
        optTransf = Zstar*0
        j = 1
        for (it in 1:nSim){
          cIdx = 1
          for (c_ in unique(Zstar)){
            pvecPost[,cIdx,j] = apply( MCMC_Obj$pvec[, MCMC_Obj$Z[,it][which(Zstar == c_)]+1,it,drop=F],1,mean)
            centroids[,cIdx,j] = apply( MCMC_Obj$muClus[, MCMC_Obj$Z[,it][which(Zstar == c_)]+1,it,drop=F],1,mean)
            coVar[,,cIdx] =  coVar[,,cIdx] +apply( MCMC_Obj$PhiClus[[it]][,,MCMC_Obj$Z[,it][which(Zstar == c_)]+1,drop=F],c(1, 2), mean)/nSim
            gamma[,cIdx,j] = apply(MCMC_Obj$gamma[,MCMC_Obj$Z[,it][which(Zstar == c_)]+1,it,drop=F],1,mean)
            optTransf[Zstar == c_] =  cIdx
            cIdx = cIdx+1
          }
          j=j+1
        }

        cen = array(apply(centroids, 2, rowMeans), dim = c(dim(centroids)[1], dim(centroids)[2]))
        pvec = array(apply(pvecPost, 2, rowMeans), dim = c(dim(pvecPost)[1], dim(pvecPost)[2]))
        gam = array(apply(gamma, 2, rowMeans), dim = c(dim(gamma)[1], dim(gamma)[2]))
        rownames(cen) = MCMC_Obj$names$UCont
        rownames(gam) = MCMC_Obj$names$Lat
      }
      rep_clust = list(Zstar = Zstar,
                       Kstar = Kstar,
                       mode = mode,
                       cenStar = centroids,
                       pvecStar = pvecPost,
                       gammaStar = gamma,
                       coVar = coVar,
                       cen = cen,
                       gamma = gam,
                       pvec = pvec)}
  }else{
    rep_clust = NULL
  }
  res = list(
    coocMat = Matrix(cooc, sparse = TRUE),
    clust = rep_clust,
    pop = pop
  )
  class(res) <- "pglmm_fit"
  return(res)
}

