#' Computes the overlap coefficient (OVL) between two cumulative distribution
#' functions corresponding to finite mixtures of two normal distributions.
#' @param mu1 Numeric vector of length 2 containing the means of the first mixture.
#' @param mu2 Numeric vector of length 2 containing the means of the second mixture.
#' @param sigma1 Numeric vector of length 2 containing the standard deviations of
#'   the first mixture.
#' @param sigma2 Numeric vector of length 2 containing the standard deviations of
#'   the second mixture.
#' @param pi_H Either a numeric scalar in [0,1] giving the mixing proportion
#'   of the first component in the first mixture, or a numeric vector of length 2
#'   with elements summing to 1.
#' @param pi_D Either a numeric scalar in [0,1] giving the mixing proportion
#'   of the first component in the second mixture, or a numeric vector of length 2
#'   with elements summing to 1.
#' @param x Numeric vector of intersection points between the two mixture
#'   densities.
#' @return
#' A numeric value corresponding to the OVL between the two
#' mixture distributions.
#' @details
#' Mixing proportions equal to 0 or 1 are allowed, in which case the corresponding
#' mixture reduces to a single normal distribution.
#' @keywords internal
#' @importFrom stats pnorm
OVL_mix = function(mu1, mu2, sigma1, sigma2, pi_H, pi_D, x) {
  suma = 0
  for (k in 1:length(x)) {
    if(pi_H[1] == 0){term1 <- 0} else{term1 <- pi_H[1] * pnorm((x[k] - mu1[1]) / sigma1[1])}
    if(1-pi_H[1] == 0){term2 <- 0}else{term2 <- (1-pi_H[1]) * pnorm((x[k] - mu1[2]) / sigma1[2]) }
    if(pi_D[1] == 0){term3 <- 0}else{term3 <- pi_D[1] * pnorm((x[k] - mu2[1]) / sigma2[1])}
    if(1-pi_D[1] == 0){term4 <- 0} else {term4 <- (1 - pi_D[1]) * pnorm((x[k] - mu2[2]) / sigma2[2])
    }
    suma <- suma + abs(term1 + term2 - term3 - term4)
  }
  OVL <- 1 - suma
  return(OVL)
}
