# LogisWeib.R
#' Logistic-Weibull Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Weibull distribution.
#'
#' The Logistic-Weibull distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Weibull distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1 - \frac{1}{{1 + {{\left( {\exp (\lambda {x^\beta }) - 1} \right)}
#' ^\alpha }}}\quad ;\,x \geqslant 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' Included functions are:
#' \itemize{
#'   \item \code{dlogis.weib()} — Density function
#'   \item \code{plogis.weib()} — Distribution function
#'   \item \code{qlogis.weib()} — Quantile function
#'   \item \code{rlogis.weib()} — Random generation
#'   \item \code{hlogis.weib()} — Hazard function
#' }
#'
#' @usage
#' dlogis.weib(x, alpha, beta, lambda, log = FALSE)
#' plogis.weib(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.weib(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.weib(n, alpha, beta, lambda)
#' hlogis.weib(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.weib}: numeric vector of (log-)densities
#'   \item \code{plogis.weib}: numeric vector of probabilities
#'   \item \code{qlogis.weib}: numeric vector of quantiles
#'   \item \code{rlogis.weib}: numeric vector of random variates
#'   \item \code{hlogis.weib}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary,A.K., & Kumar, V.(2021). 
#' The Logistic-Weibull distribution with Properties and Applications. 
#' \emph{IOSR Journal of Mathematics (IOSR-JM)}, \bold{17(1)},Ser.1, 32--41.  
#'
#' Dhungana, G.P.,  & Kumar, V.(2021). 
#' Modified Half Logistic Weibull Distribution with Statistical Properties 
#' and Applications. \emph{International Journal of Statistics and Reliability 
#' Engineering}, \bold{8(1)}, 29-39.
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dlogis.weib(x, 2.0, 0.5, 0.2)
#' plogis.weib(x, 2.0, 0.5, 0.2)
#' qlogis.weib(0.5, 2.0, 0.5, 0.2)
#' rlogis.weib(10, 2.0, 0.5, 0.2)
#' hlogis.weib(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- bladder
#' # ML estimates
#' params = list(alpha=2.4165, beta=0.5103, lambda=0.2711)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.weib, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.weib, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.weib, pfun=plogis.weib, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisWeib
#' @export
NULL

#' @rdname LogisWeib
#' @usage NULL
#' @export 
dlogis.weib <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
    u <- exp(lambda *(x[valid] ^ beta))
    num <- alpha * beta *lambda *(x[valid] ^(beta - 1.0)) * u *((u - 1.0)^(alpha - 1.0))
    deno <- (1.0 + (u - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisWeib
#' @usage NULL
#' @export 
plogis.weib <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
    u <- exp(lambda *(q[valid] ^ beta))
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0 - (1.0/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisWeib
#' @usage NULL
#' @export
qlogis.weib <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
    qtl[valid] <- ((1/lambda) * log(1.0 + ((p[valid]/(1.0 - p[valid])) ^ (1.0/alpha))))^ (1.0/beta)      
	}
    return(qtl)   
 } 

#' @rdname LogisWeib
#' @usage NULL
#' @export
rlogis.weib <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.weib(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisWeib
#' @usage NULL
#' @export
## Hazard function of Logistic-Weibull distribution
hlogis.weib <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dlogis.weib(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.weib(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
