# LogisLomax.R
#' Logistic-Lomax Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Lomax distribution.
#'
#' The Logistic-Lomax distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Lomax distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1 - \frac{1}{{1 + {{\left( {{{(1 + \beta x)}^\lambda } - 1} \right)}
#' ^\alpha }}}\quad ;\,x \geqslant 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.lomax()} — Density function
#'   \item \code{plogis.lomax()} — Distribution function
#'   \item \code{qlogis.lomax()} — Quantile function
#'   \item \code{rlogis.lomax()} — Random generation
#'   \item \code{hlogis.lomax()} — Hazard function
#' }
#'
#' @usage
#' dlogis.lomax(x, alpha, beta, lambda, log = FALSE)
#' plogis.lomax(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.lomax(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.lomax(n, alpha, beta, lambda)
#' hlogis.lomax(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.lomax}: numeric vector of (log-)densities
#'   \item \code{plogis.lomax}: numeric vector of probabilities
#'   \item \code{qlogis.lomax}: numeric vector of quantiles
#'   \item \code{rlogis.lomax}: numeric vector of random variates
#'   \item \code{hlogis.lomax}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary, A.K., & Kumar, V.(2020). 
#' The Logistic Lomax Distribution with Properties and Applications. 
#' \emph{International Journal of New Technology and Research}, 
#' \bold{6(12)}, 74--80. \doi{10.31871/IJNTR.6.12.21} 
#'
#' Shrestha, S.K., & Kumar, V. (2014). 
#' Bayesian Analysis of Extended Lomax Distribution. \emph{International 
#' Journal of Mathematical Trends and Technology (IJMTT)}, \bold{7(1)}, 33--41.  
#' \doi{10.14445/22315373/IJMTT-V7P505} 
#'  
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dlogis.lomax(x, 1.5, 0.1, 2.0)
#' plogis.lomax(x, 1.5, 0.1, 2.0)
#' qlogis.lomax(0.5, 1.5, 0.1, 2.0)
#' rlogis.lomax(10, 1.5, 0.1, 2.0)
#' hlogis.lomax(x, 1.5, 0.1, 2.0)
#' 
#' # Data
#' x <- bladder
#' # ML estimates
#' params = list(alpha=1.38027, beta=0.04451, lambda=2.80412)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.lomax, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.lomax, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'       dfun = dlogis.lomax, pfun=plogis.lomax, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisLomax
#' @export
NULL

#' @rdname LogisLomax
#' @usage NULL
#' @export  
dlogis.lomax <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
    u <- (1 + beta * x[valid])
	w <- u ^ lambda	
    num <- alpha * beta * lambda *(u^(lambda-1)) *((w-1)^(alpha-1))
    deno <- (1.0 + (w - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisLomax
#' @usage NULL
#' @export  
plogis.lomax <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- (1 + beta * q[valid]) ^ lambda 
	tmp <- 1.0 + ((u - 1) ^ alpha)
    cdf[valid] <- 1 - (1/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisLomax
#' @usage NULL
#' @export
qlogis.lomax <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- (1/beta) * (((((p[valid]/(1-p[valid])) ^ (1.0/alpha))+1) ^(1/lambda))-1)    
	}
    return(qtl)   
 } 

#' @rdname LogisLomax
#' @usage NULL
#' @export
rlogis.lomax <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.lomax(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisLomax
#' @usage NULL
#' @export
hlogis.lomax <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dlogis.lomax(x, alpha, beta, lambda, log=FALSE)
	surv <- 1.0 - plogis.lomax(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}


