plotHSMMParameters <- function(x, HSMM, obsdist, dwelldist, confint_result = NULL,
                               level = 0.95, B = 100, M = NA, include_dwell = FALSE,
                               shift = FALSE, time_structure = NULL,
                               plot_title = "HSMM Parameters Over Time",
                               overlay_data = NULL, overlay_label = "Data",
                               colors = c("black", "red", "blue", "darkgreen"),
                               save_plot = FALSE, filename = NULL,
                               width = 12, height = 8, dpi = 300,
                               verbose = TRUE, seed = NULL) {

  # Validate observation distribution
  if (!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp",
                      "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("Observation distribution not supported")
  }

  # Validate dwell time distribution
  if (!dwelldist %in% c("pois", "nbinom", "geom", "betabinom")) {
    stop("Dwell time distribution not supported")
  }

  # Extract basic dimensions
  n <- length(x)
  J <- nrow(HSMM$Pi)

  # Detect if shifted distribution is being used
  dwelldist_detected <- detectShiftedDistribution(HSMM$dwellparameters, dwelldist, J)

  # Decode most likely state sequence
  decoded_states <- globaldecodeHSMM(
    x = x,
    M = M,
    HSMM = HSMM,
    obsdist = obsdist,
    dwelldist = dwelldist_detected
  )

  # Extract parameters and get distribution-specific info
  obspar <- HSMM$observationparameters
  dwellpar <- HSMM$dwellparameters
  obs_param_info <- getParameterInfo(obsdist, J)

  # Get dwell parameter info if needed
  if (include_dwell) {
    include_shift <- FALSE
    if (startsWith(dwelldist_detected, "shift") && !is.null(dwellpar$shift)) {
      if (length(unique(dwellpar$shift)) > 1) {
        include_shift <- TRUE
      }
    }

    dwell_param_info <- getDwellParameterInfo(dwelldist_detected, J, include_shift)
  } else {
    dwell_param_info <- list(count = 0, names = character(0), labels = character(0))
  }

  # Compute confidence intervals if not provided
  if (is.null(confint_result)) {
    if (verbose) message("Computing confidence intervals...")
    confint_result <- confintHSMM(x, HSMM, obsdist, dwelldist_detected,
                                  level = level, B = B, M = M, shift = shift,
                                  verbose = verbose, seed = seed)
  }

  # Set up time axis information
  time_info <- createTimeInfo(n, time_structure)

  # Create parameter time series based on decoded states
  obs_param_series <- createParameterTimeSeries(obspar, decoded_states, obs_param_info, n, J)

  if (include_dwell) {
    dwell_param_series <- createDwellParameterTimeSeries(dwellpar, decoded_states, dwell_param_info, n, J)
    dwell_ci_series <- createCITimeSeriesHSMM(confint_result$dwellpar_CI, decoded_states, dwell_param_info, n, J)
  } else {
    dwell_param_series <- list()
    dwell_ci_series <- list()
  }

  # Create confidence interval time series
  obs_ci_series <- createCITimeSeriesHSMM(confint_result$obspar_CI, decoded_states, obs_param_info, n, J)

  # Initialize graphics device for saving if requested
  if (save_plot) {
    if (is.null(filename)) {
      stop("filename must be specified when save_plot = TRUE")
    }
    png(filename, width = width, height = height, units = "in", res = dpi)
    on.exit(dev.off(), add = TRUE)
  }

  # Save and restore graphics parameters
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Generate the parameter plots
  createThreeTierHSMMPlot(
    obs_param_series = obs_param_series,
    obs_ci_series = obs_ci_series,
    obs_param_info = obs_param_info,
    dwell_param_series = dwell_param_series,
    dwell_ci_series = dwell_ci_series,
    dwell_param_info = dwell_param_info,
    time_info = time_info,
    plot_title = plot_title,
    overlay_data = overlay_data,
    overlay_label = overlay_label,
    colors = colors,
    include_dwell = include_dwell
  )

  if (save_plot && verbose) {
    message("Plot saved to: ", filename)
  }

  invisible(list(
    obs_param_series = obs_param_series,
    obs_ci_series = obs_ci_series,
    dwell_param_series = dwell_param_series,
    dwell_ci_series = dwell_ci_series,
    time_info = time_info,
    decoded_states = decoded_states
  ))
}

detectShiftedDistribution <- function(dwellpar, dwelldist, J) {
  if (!is.null(dwellpar$shift)) {
    if (all(dwellpar$shift == 1, na.rm = TRUE)) {
      return(dwelldist)
    } else {
      if (!startsWith(dwelldist, "shift")) {
        return(paste0("shift", dwelldist))
      } else {
        return(dwelldist)
      }
    }
  } else {
    return(dwelldist)
  }
}

createTimeInfo <- function(n, time_structure = NULL) {

  # Default to simple observation numbering if no structure provided
  if (is.null(time_structure)) {
    return(list(
      labels = 1:n,
      unit = "observation",
      conversion_factor = 1,
      unit_name = "observation",
      x_label = "Time"
    ))
  }

  # Validate required fields based on time structure type
  required_fields <- c("unit", "observations_per_unit")
  if (time_structure$unit == "custom") {
    required_fields <- c("conversion_factor", "unit_name")
  }

  missing_fields <- setdiff(required_fields, names(time_structure))
  if (length(missing_fields) > 0) {
    stop("Missing required fields in time_structure: ", paste(missing_fields, collapse = ", "))
  }

  # Set conversion factor and unit names
  if (time_structure$unit == "custom") {
    conversion_factor <- time_structure$conversion_factor
    unit_name <- time_structure$unit_name
    x_label <- paste("Time (", unit_name, "s)", sep = "")
  } else {
    conversion_factor <- time_structure$observations_per_unit
    unit_name <- time_structure$unit

    # Create appropriate axis labels for common units
    if (unit_name == "year") {
      x_label <- "Time (years)"
    } else if (unit_name == "day") {
      x_label <- "Time (days)"
    } else if (unit_name == "hour") {
      x_label <- "Time (hours)"
    } else {
      x_label <- paste("Time (", unit_name, "s)", sep = "")
    }
  }

  # Calculate time duration and handle start/end points
  duration <- n / conversion_factor

  has_start <- !is.null(time_structure$start_point)
  has_end <- !is.null(time_structure$end_point)

  if (has_start && has_end) {
    start_point <- time_structure$start_point
    end_point <- time_structure$end_point
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (has_start && !has_end) {
    start_point <- time_structure$start_point
    end_point <- start_point + duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (!has_start && has_end) {
    end_point <- time_structure$end_point
    start_point <- end_point - duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else {
    start_point <- 0
    end_point <- duration
    time_labels <- seq(start_point, end_point, length.out = n)
  }

  return(list(
    labels = time_labels,
    unit = ifelse(time_structure$unit == "custom", "custom", time_structure$unit),
    conversion_factor = conversion_factor,
    unit_name = unit_name,
    x_label = x_label,
    duration = duration,
    start_point = start_point,
    end_point = end_point
  ))
}

createThreeTierHSMMPlot <- function(obs_param_series, obs_ci_series, obs_param_info,
                                    dwell_param_series, dwell_ci_series, dwell_param_info,
                                    time_info, plot_title, overlay_data, overlay_label, colors,
                                    include_dwell = FALSE) {

  n_obs_params <- obs_param_info$count
  n_dwell_params <- ifelse(include_dwell, dwell_param_info$count, 0)
  time_labels <- time_info$labels
  n <- length(time_labels)

  # Set up panel layout based on whether dwell parameters are included
  if (include_dwell) {
    total_rows <- 6
    total_plots <- n_obs_params + n_dwell_params

    # Determine column structure
    if (n_dwell_params <= 2) {
      max_cols <- max(n_obs_params, n_dwell_params)
    } else {
      max_cols <- max(n_obs_params, 2)
    }

    # Create layout matrix: observation parameters occupy rows 1-4, dwell parameters occupy rows 5-6
    layout_matrix <- matrix(0, nrow = total_rows, ncol = max_cols)

    # Fill observation parameter positions (rows 1-4)
    for (j in 1:n_obs_params) {
      for (i in 1:4) {
        layout_matrix[i, j] <- j
      }
    }

    # Fill remaining columns with last observation parameter if needed
    if (n_obs_params < max_cols) {
      for (j in (n_obs_params + 1):max_cols) {
        for (i in 1:4) {
          layout_matrix[i, j] <- n_obs_params
        }
      }
    }

    dwell_start_num <- n_obs_params + 1

    # Fill dwell parameter positions (rows 5-6)
    if (n_dwell_params == 1) {
      for (i in 5:6) {
        for (j in 1:max_cols) {
          layout_matrix[i, j] <- dwell_start_num
        }
      }
    } else if (n_dwell_params == 2) {
      for (i in 5:6) {
        layout_matrix[i, 1] <- dwell_start_num
        layout_matrix[i, 2] <- dwell_start_num + 1
      }
    } else if (n_dwell_params == 3 || n_dwell_params == 4) {
      layout_matrix[5, 1] <- dwell_start_num
      layout_matrix[5, 2] <- dwell_start_num + 1
      layout_matrix[6, 1] <- dwell_start_num + 2
      if (n_dwell_params == 4) {
        layout_matrix[6, 2] <- dwell_start_num + 3
      } else {
        layout_matrix[6, 2] <- dwell_start_num + 2
      }
    } else {
      stop("More than 4 dwell parameters not supported in current layout")
    }

    # Validate layout matrix
    if (any(layout_matrix == 0)) {
      stop("Layout matrix contains zeros")
    }
    if (max(layout_matrix) > total_plots) {
      stop("Layout matrix contains invalid plot numbers")
    }

  } else {
    # Simple single-row layout for observation parameters only
    max_cols <- n_obs_params
    layout_matrix <- matrix(0, nrow = 1, ncol = max_cols)

    for (j in 1:n_obs_params) {
      layout_matrix[1, j] <- j
    }

    total_plots <- n_obs_params
  }

  layout(layout_matrix)

  # Plot observation parameters
  for (p in 1:n_obs_params) {
    param_name <- obs_param_info$names[p]
    param_label <- obs_param_info$labels[p]
    series <- obs_param_series[[param_name]]

    # Set y-axis limits to accommodate parameter values and CIs
    if (param_name %in% names(obs_ci_series)) {
      ylim <- range(c(series, obs_ci_series[[param_name]]$lower, obs_ci_series[[param_name]]$upper), na.rm = TRUE)
    } else {
      ylim <- range(series, na.rm = TRUE)
    }

    # Adjust y-axis limits for overlay data if present
    if (!is.null(overlay_data) && length(overlay_data) == length(time_labels)) {
      ylim_range <- diff(ylim)
      ylim[1] <- ylim[1] - 0.5 * ylim_range
      ylim[2] <- ylim[2] + 0.05 * ylim_range
    } else {
      ylim_range <- diff(ylim)
      ylim[1] <- ylim[1] - 0.05 * ylim_range
      ylim[2] <- ylim[2] + 0.05 * ylim_range
    }

    # Set plot margins and x-axis based on whether dwell parameters are included
    if (include_dwell) {
      par(mar = c(2, 4.5, 2.5, 3.5))
      xaxt_val <- "n"
      xlab_val <- ""
    } else {
      par(mar = c(4.5, 4.5, 2.5, 3.5))
      xaxt_val <- "n"
      xlab_val <- time_info$x_label
    }

    # Plot main parameter time series
    plot(time_labels, series, type = "l", lwd = 2, col = colors[1],
         xlab = xlab_val, ylab = param_label,
         main = paste("(", letters[p], ") ", param_label, sep = ""),
         ylim = ylim, cex.lab = 1.3, cex.axis = 1.2, xaxt = xaxt_val)

    addCustomXAxis(time_labels, time_info, show_label = !include_dwell)

    # Add confidence intervals if available
    if (param_name %in% names(obs_ci_series)) {
      lines(time_labels, obs_ci_series[[param_name]]$lower, lty = "dashed", col = colors[2])
      lines(time_labels, obs_ci_series[[param_name]]$upper, lty = "dashed", col = colors[2])
    }

    # Add overlay data if provided
    if (!is.null(overlay_data) && length(overlay_data) == length(time_labels)) {
      addOverlayData(time_labels, overlay_data, overlay_label, param_name,
                     param_label, ylim, p, n_obs_params)
    }
  }

  # Plot dwell parameters if requested
  if (include_dwell && n_dwell_params > 0) {
    for (p in 1:n_dwell_params) {
      param_name <- dwell_param_info$names[p]
      param_label <- dwell_param_info$labels[p]
      series <- dwell_param_series[[param_name]]

      # Set y-axis limits to accommodate parameter values and CIs
      if (param_name %in% names(dwell_ci_series)) {
        ylim <- range(c(series, dwell_ci_series[[param_name]]$lower, dwell_ci_series[[param_name]]$upper), na.rm = TRUE)
      } else {
        ylim <- range(series, na.rm = TRUE)
      }

      ylim_range <- diff(ylim)
      ylim[1] <- ylim[1] - 0.05 * ylim_range
      ylim[2] <- ylim[2] + 0.05 * ylim_range

      par(mar = c(4.5, 4.5, 1.5, 1.5))

      # Plot dwell parameter time series
      plot(time_labels, series, type = "l", lwd = 2, col = colors[3],
           xlab = time_info$x_label, ylab = param_label,
           ylim = ylim, cex.lab = 1.2, cex.axis = 1.1, xaxt = "n")

      addCustomXAxis(time_labels, time_info, show_label = TRUE)

      # Add confidence intervals if available
      if (param_name %in% names(dwell_ci_series)) {
        lines(time_labels, dwell_ci_series[[param_name]]$lower, lty = "dashed", col = colors[4])
        lines(time_labels, dwell_ci_series[[param_name]]$upper, lty = "dashed", col = colors[4])
      }
    }
  }

  layout(1)
}

addCustomXAxis <- function(time_labels, time_info, show_label = TRUE) {

  # Special formatting for year-based time axes
  if (time_info$unit == "year" && !is.null(time_info$start_point) && time_info$start_point > 1800) {

    year_range <- range(time_labels)
    year_span <- diff(year_range)

    # Choose tick spacing based on year span
    if (year_span <= 2) {
      tick_positions <- seq(ceiling(year_range[1] * 4) / 4, floor(year_range[2] * 4) / 4, by = 0.25)
      tick_labels <- formatYearLabels(tick_positions, format = "month")
    } else if (year_span <= 10) {
      tick_positions <- seq(ceiling(year_range[1]), floor(year_range[2]), by = 1)
      tick_labels <- formatYearLabels(tick_positions, format = "year")
    } else if (year_span <= 50) {
      step <- ceiling(year_span / 8)
      tick_positions <- seq(ceiling(year_range[1] / step) * step, floor(year_range[2] / step) * step, by = step)
      tick_labels <- formatYearLabels(tick_positions, format = "year")
    } else {
      step <- ceiling(year_span / 10)
      tick_positions <- seq(ceiling(year_range[1] / step) * step, floor(year_range[2] / step) * step, by = step)
      tick_labels <- formatYearLabels(tick_positions, format = "year")
    }

    axis(1, at = tick_positions, labels = tick_labels, cex.axis = 1.2)

  } else {
    axis(1, cex.axis = 1.2)
  }
}

formatYearLabels <- function(year_values, format = "year") {

  # Format labels as "Mon YYYY" for quarterly/monthly data
  if (format == "month") {
    labels <- character(length(year_values))
    for (i in seq_along(year_values)) {
      year <- floor(year_values[i])
      month_frac <- year_values[i] - year
      month <- round(month_frac * 12) + 1
      if (month > 12) {
        month <- 12
        year <- year + 1
      }

      month_names <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun",
                       "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
      labels[i] <- paste0(month_names[month], " ", year)
    }
    return(labels)

  } else {
    # Format as simple year for annual data
    return(as.character(round(year_values)))
  }
}

addOverlayData <- function(time_labels, overlay_data, overlay_label, param_name,
                           param_label, ylim, p, n_params) {

  # Choose scaling factor based on parameter type
  if (param_name == "loc" || grepl("location", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  } else if (param_name == "scale" || grepl("scale", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  } else if (param_name == "shape" || grepl("shape", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.32)
  } else {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  }

  # Add overlay line
  lines(time_labels, overlay_scaled, col = "grey30", lwd = 1)

  # Create right-hand axis with original overlay scale
  overlay_range <- range(overlay_data, na.rm = TRUE)
  overlay_ticks <- pretty(overlay_range, n = 3)

  # Scale tick positions to match overlay data scaling
  if (param_name == "loc" || grepl("location", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  } else if (param_name == "scale" || grepl("scale", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  } else if (param_name == "shape" || grepl("shape", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.32)
  } else {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  }

  # Add right-hand axis
  axis(4, at = overlay_scaled_ticks, labels = overlay_ticks,
       cex.axis = 1.2, col = "darkgrey", col.axis = "darkgrey")

  # Add overlay label to rightmost subplot
  if (p == n_params) {
    mtext(overlay_label, side = 4, line = 2.2, cex = 1.3,
          las = 0, adj = 0, col = "darkgrey")
  }
}

getParameterInfo <- function(obsdist, J) {
  param_info <- switch(obsdist,
                       "pois" = list(
                         names = "lambda",
                         labels = "Poisson rate",
                         count = 1,
                         transform = exp
                       ),
                       "norm" = list(
                         names = c("mean", "sd"),
                         labels = c("Normal mean", "Normal SD"),
                         count = 2,
                         transform = list(identity, exp)
                       ),
                       "weibull" = list(
                         names = c("shape", "scale"),
                         labels = c("Weibull shape", "Weibull scale"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "zip" = list(
                         names = c("pi", "lambda"),
                         labels = c("ZIP zero probability", "ZIP rate"),
                         count = 2,
                         transform = list(plogis, exp)
                       ),
                       "nbinom" = list(
                         names = c("size", "mu"),
                         labels = c("NB size", "NB mean"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "zinb" = list(
                         names = c("pi", "size", "mu"),
                         labels = c("ZINB zero probability", "ZINB size", "ZINB mean"),
                         count = 3,
                         transform = list(plogis, exp, exp)
                       ),
                       "exp" = list(
                         names = "rate",
                         labels = "Exponential rate",
                         count = 1,
                         transform = exp
                       ),
                       "gamma" = list(
                         names = c("shape", "rate"),
                         labels = c("Gamma shape", "Gamma rate"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "lnorm" = list(
                         names = c("meanlog", "sdlog"),
                         labels = c("Log-normal mean", "Log-normal SD"),
                         count = 2,
                         transform = list(identity, exp)
                       ),
                       "gev" = list(
                         names = c("loc", "scale", "shape"),
                         labels = c("GEV location", "GEV scale", "GEV shape"),
                         count = 3,
                         transform = list(identity, exp, identity)
                       ),
                       "ZInormal" = list(
                         names = c("pi", "mean", "sd"),
                         labels = c("ZI-Normal zero probability", "ZI-Normal mean", "ZI-Normal SD"),
                         count = 3,
                         transform = list(plogis, identity, exp)
                       ),
                       "ZIgamma" = list(
                         names = c("pi", "shape", "rate"),
                         labels = c("ZI-Gamma zero probability", "ZI-Gamma shape", "ZI-Gamma rate"),
                         count = 3,
                         transform = list(plogis, exp, exp)
                       )
  )

  return(param_info)
}

getDwellParameterInfo <- function(dwelldist, J, include_shift = FALSE) {
  param_info <- switch(dwelldist,
                       "pois" = list(
                         names = "lambda",
                         labels = "Poisson dwell rate",
                         count = 1,
                         transform = exp
                       ),
                       "nbinom" = list(
                         names = c("size", "mu"),
                         labels = c("NB dwell size", "NB dwell mean"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "geom" = list(
                         names = "prob",
                         labels = "Geometric dwell probability",
                         count = 1,
                         transform = plogis
                       ),
                       "betabinom" = list(
                         names = c("size", "alpha", "beta"),
                         labels = c("Beta-binomial dwell size", "Beta-binomial alpha", "Beta-binomial beta"),
                         count = 3,
                         transform = list(exp, exp, exp)
                       ),
                       "shiftpois" = list(
                         names = if(include_shift) c("lambda", "shift") else "lambda",
                         labels = if(include_shift) c("Shifted Poisson dwell rate", "Poisson shift") else "Shifted Poisson dwell rate",
                         count = if(include_shift) 2 else 1,
                         transform = if(include_shift) list(exp, identity) else exp
                       ),
                       "shiftgeom" = list(
                         names = if(include_shift) c("prob", "shift") else "prob",
                         labels = if(include_shift) c("Shifted geometric dwell probability", "Geometric shift") else "Shifted geometric dwell probability",
                         count = if(include_shift) 2 else 1,
                         transform = if(include_shift) list(plogis, identity) else plogis
                       ),
                       "shiftnbinom" = list(
                         names = if(include_shift) c("size", "mu", "shift") else c("size", "mu"),
                         labels = if(include_shift) c("Shifted NB dwell size", "Shifted NB dwell mean", "NB shift") else c("Shifted NB dwell size", "Shifted NB dwell mean"),
                         count = if(include_shift) 3 else 2,
                         transform = if(include_shift) list(exp, exp, identity) else list(exp, exp)
                       ),
                       "shiftbetabinom" = list(
                         names = if(include_shift) c("size", "alpha", "beta", "shift") else c("size", "alpha", "beta"),
                         labels = if(include_shift) c("Shifted Beta-binomial dwell size", "Shifted Beta-binomial alpha",
                                                      "Shifted Beta-binomial beta", "Beta-binomial shift") else c("Shifted Beta-binomial dwell size", "Shifted Beta-binomial alpha",
                                                                                                                  "Shifted Beta-binomial beta"),
                         count = if(include_shift) 4 else 3,
                         transform = if(include_shift) list(exp, exp, exp, identity) else list(exp, exp, exp)
                       )
  )

  return(param_info)
}

createParameterTimeSeries <- function(obspar, decoded_states, param_info, n, J) {
  param_series <- list()

  # Create time series for each parameter
  for (p in 1:param_info$count) {
    param_name <- param_info$names[p]
    param_values <- obspar[[param_name]]

    # Map state-specific parameters to time points
    series <- numeric(n)
    for (t in 1:n) {
      series[t] <- param_values[decoded_states[t]]
    }

    param_series[[param_name]] <- series
  }

  return(param_series)
}

createDwellParameterTimeSeries <- function(dwellpar, decoded_states, param_info, n, J) {
  param_series <- list()

  # Create time series for each dwell parameter
  for (p in 1:param_info$count) {
    param_name <- param_info$names[p]
    param_values <- dwellpar[[param_name]]

    # Map state-specific parameters to time points
    series <- numeric(n)
    for (t in 1:n) {
      series[t] <- param_values[decoded_states[t]]
    }

    param_series[[param_name]] <- series
  }

  return(param_series)
}

createCITimeSeriesHSMM <- function(confint_result, decoded_states, param_info, n, J) {
  ci_series <- list()

  if (is.null(confint_result)) {
    return(ci_series)
  }

  # Add Parameter column if not present
  if (!"Parameter" %in% names(confint_result)) {
    if (!is.null(rownames(confint_result))) {
      confint_result$Parameter <- rownames(confint_result)
    } else {
      warning("No Parameter column or rownames found in confint_result")
      return(ci_series)
    }
  }

  # Create CI time series for each parameter
  for (p in 1:param_info$count) {
    param_name <- param_info$names[p]

    # Find parameter rows in confidence interval results
    param_pattern <- paste0("^", param_name, "[0-9]+$")
    param_rows <- grep(param_pattern, confint_result$Parameter, ignore.case = FALSE)

    if (length(param_rows) == J) {
      param_names <- confint_result$Parameter[param_rows]
      state_numbers <- as.numeric(gsub(param_name, "", param_names, ignore.case = TRUE))
      sorted_indices <- order(state_numbers)
      param_rows <- param_rows[sorted_indices]

      lower_values <- confint_result$Lower[param_rows]
      upper_values <- confint_result$Upper[param_rows]

      # Map state-specific CIs to time points
      lower_series <- numeric(n)
      upper_series <- numeric(n)

      for (t in 1:n) {
        lower_series[t] <- lower_values[decoded_states[t]]
        upper_series[t] <- upper_values[decoded_states[t]]
      }

      ci_series[[param_name]] <- list(
        lower = lower_series,
        upper = upper_series
      )
    }
  }

  return(ci_series)
}

scale_overlay_to_bottom <- function(x, target_range, scale_factor = 0.28) {
  # Scale overlay data to bottom portion of y-axis range
  x_range <- range(x, na.rm = TRUE)
  x_normalized <- (x - x_range[1]) / diff(x_range)
  bottom_start <- target_range[1]
  bottom_height <- diff(target_range) * scale_factor
  x_scaled <- x_normalized * bottom_height + bottom_start
  return(x_scaled)
}

`%||%` <- function(x, y) {
  if (is.null(x)) y else x
}
