

runEM <- function(x, param, max_iter = 100, tol = 1e-5) {
  
 
  data <- as.matrix(x)
  N <- nrow(data)
  P <- ncol(data)
  
 
  pi_k  <- param$pi_k
  mu    <- param$mu
  sigma <- param$sigma
  
  K <- length(pi_k)
  gamma <- matrix(0, N, K)
  
  log_likelihood_old <- -Inf
  
  
 
  for (iter in 1:max_iter) {
    

    for (i in 1:K) {
      gamma[, i] <- pi_k[i] * safe_dmv(data, mu = mu[i,], sigma = sigma[,,i])
    }
    

    row_den <- rowSums(gamma)
    row_den <- pmax(row_den, .Machine$double.eps)
    
    
    gamma <- gamma / row_den
    

    Nk <- colSums(gamma)
    pi_k <- Nk / N
    
    for (i in 1:K) {

      mu[i, ] <- colSums(data * gamma[, i]) / Nk[i]
      
      
      centered <- sweep(data, 2, mu[i, ])
      weighted <- centered * gamma[, i]
      sigma[,,i] <- t(centered) %*% weighted / Nk[i]
      
     
      sigma[,,i] <- sigma[,,i] + diag(1e-6, P)
    }
    
    

    log_likelihood_new <- sum(log(row_den))
    

    diff  <- abs(log_likelihood_new - log_likelihood_old)
    scale <- max(abs(log_likelihood_old), .Machine$double.eps)
    ratio <- diff / scale
    
   
    if (!is.nan(ratio) && !is.infinite(ratio) && ratio < tol) {
      break
    }
    
    log_likelihood_old <- log_likelihood_new
  }
  
  
 
  cluster_labels <- apply(gamma, 1, which.max)
  

  Z <- gamma
  

  BIC_value <- function(logL, n, p, k) {
    M <- k*p + k*(p*(p+1)/2) + (k - 1)
    -2 * logL + M * log(n)
  }
  
  BIC <- BIC_value(log_likelihood_new, N, P, K)
  
  
 
  new_param <- list(
    pi_k = pi_k,
    mu    = mu,
    sigma = sigma
  )
  

  out <- list(
    BIC = BIC,
    param = new_param,
    cluster = cluster_labels,
    Z = Z
  )
  class(out) <- "gmminit_fit"
  return(out)
  
}


#' @export
print.gmminit_fit <- function(x, ...) {
  if (!is.null(x$Z)) {
    if (is.matrix(x$Z)) {
      x$Z <- sprintf("<hidden> (%dx%d; use x$Z)", nrow(x$Z), ncol(x$Z))
    } else {
      x$Z <- "<hidden> (use x$Z)"
    }
  }
  NextMethod("print")   # <-- key fix
  invisible(x)
}





