#' Summarize elements of a numerical vector that occur unusually often
#'
#' Applies the three-sigma edit rule to the frequencies of distinct values
#' of a numerical vector, finding those that occur unusually often and
#' identifying them either by record number or an associated identifying
#' characteristic specified by `label`.  Inliers often represent data
#' values that are incorrect but consistent with the overall data distribution,
#' as in the case of numerically-coded disguised missing data
#'
#' @param x numerical vector in which inliers are to be detected
#' @param label optional identifying tag for inliers (default NULL gives
#' an index into the elements of `x` declared inliers)
#' @param labelName optional name for the `label` column, if specified
#' (default NULL labels this column as Case)
#' @param t detection threshold for the three-sigma edit rule applied to
#' record counts (default value 3)
#'
#' @return  Data frame with one row for each inlier detected and two columns:
#'   * Record (or `labelName` value) identifying or characterizing each inlier
#'   * Value the numerical value that occurs unusually often
#'
#' Note that this data frame is empty (0 rows) if no inliers are detected
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 100)
#' x[45:54] <- 0
#' SummarizeInliers(x)
SummarizeInliers <- function(x, label = NULL, labelName = NULL, t = 3){
  #
  stopifnot("x must be numeric"= is.numeric(x))
  stopifnot("t must be non-negative"= t >= 0)
  nonMiss <- length(which(!is.na(x)))
  stopifnot("No non-missing values to characterize"= nonMiss > 0)
  #
  inIndex <- FindInliers(x, t)
  nIn <- length(inIndex)
  if (nIn == 0){
    outFrame <- data.frame(Record = numeric(0), Value = numeric(0))
  } else {
    if (is.null(label)){
      label <- seq(1, length(x))
      labelName <- "Record"
    } else {
      if (is.null(labelName)){
        labelName <- "Case"
      }
    }
    outFrame <- data.frame(label = label[inIndex],
                           x = x[inIndex])
    colnames(outFrame) <- c(labelName, "Value")
  }
  return(outFrame)
}
