
#' Compuute outlier limits by three methods
#'
#' Compute upper and lower outlier limits by three detection rules:
#' the 3-sigma edit rule, the Hampel identifier, or the boxplot rule
#'
#' @param x numerical vector in which outliers are to be detected
#' @param method single character specifying the outlier rule (T, H, or B)
#' @param t threshold parameter (default NULL, gives 3 for T and H rules, 1.5 for B rule)
#'
#' @return named numerical vector with these 4 elements:
#'   * nRec the number of elements in `x`
#'   * nonMiss the number of non-missing elements in `x`
#'   * loLim the lower outlier threshold for `x` elements
#'   * upLim the upper outlier threshold for `x` elements
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 100)
#' x[1:10] <- 10
#' ComputeOutlierLimits(x, "T")
#' ComputeOutlierLimits(x, "H")
#' ComputeOutlierLimits(x, "B")
ComputeOutlierLimits <- function(x, method, t = NULL){
  #
  stopifnot("x must be numeric"= is.numeric(x))
  validMethods <- c("T", "H", "B")
  stopifnot("Invalid method argument"= method %in% validMethods)
  stopifnot("t must be NULL or non-negative"= (is.null(t) | t >= 0))
  #
  nRec <- length(x)
  if (nRec == 0){
    nonMiss <- 0
    loLim <- 0
    upLim <- 0
  } else {
    y <- x[which(!is.na(x))]
    nonMiss <- length(y)
    if (nonMiss == 0){
      loLim <- 0
      upLim <- 0
    } else {
      if (method == "T"){
        mu <- mean(y)
        sig <- stats::sd(y)
        t <- ifelse(is.null(t), 3, t)
        loLim <- mu - t * sig
        upLim <- mu + t * sig
      } else {
        if (method == "H"){
          mu <- stats::median(y)
          sig <- stats::mad(y)
          t <- ifelse(is.null(t), 3, t)
          loLim <- mu - t * sig
          upLim <- mu + t * sig
        } else {
          qtls <- as.numeric(stats::quantile(y, probs = c(0.25, 0.75)))
          iqd <- qtls[2] - qtls[1]
          t <- ifelse(is.null(t), 1.5, t)
          loLim <- qtls[1] - t * iqd
          upLim <- qtls[2] + t * iqd
        }
      }
    }
  }
  outVec <- c(nRec, nonMiss, loLim, upLim)
  names(outVec) <- c("nRec", "nonMiss", "loLim", "upLim")
  return(outVec)
}


