#' Compare Code Saturation Across Threshold Sets
#'
#' @description
#' This function compares code saturation results from a code summary table
#' (typically generated by [create_code_summary()]) against one or more
#' threshold sets that define what constitutes "saturation" based on the
#' number of excerpts (`count`) and proportion of media titles (`prop_media_titles`)
#' where each code appears.
#'
#' Optionally, the function can produce a faceted bar plot showing which codes
#' meet each set of thresholds, with metrics plotted as counts, proportions, or both.
#'
#' @param code_summary A data frame or tibble, typically produced by
#'   [create_code_summary()], containing columns:
#'   \code{code}, \code{count}, \code{n_media_titles}, and \code{prop_media_titles}.
#' @param thresholds_list A named list of threshold sets. Each element should
#'   be a list with numeric elements \code{code_count} and \code{prop_media_title}.
#'   For example:
#'   \code{list(Liberal = list(code_count = 3, prop_media_title = 0.2),
#'   Strict = list(code_count = 5, prop_media_title = 0.5))}.
#' @param output_type Character string specifying the output type:
#'   either \code{"tibble"} (default) or \code{"kable"} for a formatted table.
#' @param plot Logical; if \code{TRUE}, generates a ggplot visualization
#'   of which codes meet each threshold set.
#' @param plot_metric Character string specifying which metric to plot:
#'   \itemize{
#'     \item \code{"count"} — show excerpt frequencies only.
#'     \item \code{"prop"} — show proportions of media titles only.
#'     \item \code{"both"} — show both metrics with dual y-axes
#'       (counts on the bottom axis, proportions on the top axis).
#'   }
#'
#' @return
#' If \code{plot = FALSE}, returns either a tibble or kable table of results.
#' If \code{plot = TRUE}, returns a list with:
#' \itemize{
#'   \item \code{results}: the tibble or kable table with logical columns indicating which codes meet each threshold set.
#'   \item \code{plot}: a ggplot2 object visualizing saturation across threshold sets.
#' }
#'
#' @details
#' Each threshold set is applied independently. A code is considered to meet a
#' given threshold set if both its excerpt count and proportion of media titles
#' are greater than or equal to the respective thresholds.
#'
#' @importFrom dplyr n_distinct select all_of filter mutate group_by summarise desc arrange
#' @importFrom tidyr pivot_longer
#' @importFrom ggplot2 ggplot aes geom_col coord_flip facet_wrap labs theme_minimal scale_y_continuous sec_axis
#' @importFrom stats reorder
#' @importFrom knitr kable
#'
#' @examples
#' # Example data: excerpts with coded logical columns
#' set.seed(123)
#' excerpts <- data.frame(
#'   media_title = rep(paste0("Interview_", 1:5), each = 3),
#'   code_A = sample(c(TRUE, FALSE), 15, replace = TRUE),
#'   code_B = sample(c(TRUE, FALSE), 15, replace = TRUE),
#'   code_C = sample(c(TRUE, FALSE), 15, replace = TRUE)
#' )
#'
#' # Create a code summary table (from your package function)
#' code_summary <- create_code_summary(excerpts, output_type = "tibble")
#'
#' # Define two saturation threshold sets
#' thresholds_list <- list(
#'   Liberal = list(code_count = 3, prop_media_title = 0.2),
#'   Strict  = list(code_count = 5, prop_media_title = 0.5)
#' )
#'
#' # Compare saturation (table only)
#' compare_saturation(code_summary, thresholds_list)
#'
#' # Compare and plot using proportions
#' res <- compare_saturation(code_summary, thresholds_list, plot = TRUE, plot_metric = "prop")
#' res$plot
#'
#' # Compare and plot both metrics with dual y-axes
#' res2 <- compare_saturation(code_summary, thresholds_list, plot = TRUE, plot_metric = "both")
#' res2$plot
#'
#' @export
compare_saturation <- function(code_summary,
                               thresholds_list,
                               output_type = c("tibble", "kable"),
                               plot = FALSE,
                               plot_metric = c("count", "prop", "both")) {
  output_type <- match.arg(output_type)
  plot_metric <- match.arg(plot_metric)

  # --- Validate inputs ---
  required_cols <- c("code", "count", "n_media_titles", "prop_media_titles")
  missing_cols <- setdiff(required_cols, names(code_summary))
  if (length(missing_cols) > 0) {
    stop("`code_summary` must contain the following columns: ",
         paste(missing_cols, collapse = ", "))
  }

  # --- Apply thresholds ---
  compare_df <- code_summary
  for (set_name in names(thresholds_list)) {
    rules <- thresholds_list[[set_name]]
    compare_df[[set_name]] <-
      compare_df$count >= rules$code_count &
      compare_df$prop_media_titles >= rules$prop_media_title
  }

  # --- Output table ---
  if (output_type == "kable") {
    comparison_output <- knitr::kable(compare_df,
                                      caption = "Saturation Comparison")
  } else {
    comparison_output <- compare_df
  }

  # --- Optional plotting ---
  if (plot) {
    set_cols <- names(thresholds_list)

    plot_data <- compare_df %>%
      dplyr::select(code, count, prop_media_titles, dplyr::all_of(set_cols)) %>%
      tidyr::pivot_longer(
        cols = dplyr::all_of(set_cols),
        names_to = "Set",
        values_to = "Meets"
      ) %>%
      dplyr::filter(Meets == TRUE)

    # ---- Choose metric ----
    if (plot_metric == "count") {
      p <- ggplot2::ggplot(
        plot_data,
        ggplot2::aes(x = reorder(code, count),
                     y = count,
                     fill = Set)
      ) +
        ggplot2::geom_col() +
        ggplot2::coord_flip() +
        ggplot2::facet_wrap(~ Set, ncol = 1, scales = "free_y") +
        ggplot2::labs(
          x = "Code",
          y = "Excerpt Frequency",
          title = "Code Saturation Comparison by Threshold Set (Counts)"
        ) +
        ggplot2::theme_minimal()

    } else if (plot_metric == "prop") {
      p <- ggplot2::ggplot(
        plot_data,
        ggplot2::aes(x = reorder(code, prop_media_titles),
                     y = prop_media_titles,
                     fill = Set)
      ) +
        ggplot2::geom_col() +
        ggplot2::coord_flip() +
        ggplot2::facet_wrap(~ Set, ncol = 1, scales = "free_y") +
        ggplot2::labs(
          x = "Code",
          y = "Proportion of Media Titles",
          title = "Code Saturation Comparison by Threshold Set (Proportions)"
        ) +
        ggplot2::theme_minimal()

    } else if (plot_metric == "both") {
      scale_factor <- max(plot_data$count, na.rm = TRUE) /
        max(plot_data$prop_media_titles, na.rm = TRUE)

      p <- ggplot2::ggplot(
        plot_data,
        ggplot2::aes(x = reorder(code, count),
                     y = count,
                     fill = Set)
      ) +
        ggplot2::geom_col() +
        ggplot2::coord_flip() +
        ggplot2::facet_wrap(~ Set, ncol = 1, scales = "free_y") +
        ggplot2::scale_y_continuous(
          name = "Excerpt Frequency",
          sec.axis = ggplot2::sec_axis(~ . / scale_factor,
                                       name = "Proportion of Media Titles")
        ) +
        ggplot2::labs(
          x = "Code",
          title = "Code Saturation Comparison by Threshold Set (Counts + Proportions)"
        ) +
        ggplot2::theme_minimal()
    }

    return(list(results = comparison_output, plot = p))
  }

  return(comparison_output)
}
