% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clust_cp.R
\name{clust_cp}
\alias{clust_cp}
\title{Clustering time dependent observations with common change points.}
\usage{
clust_cp(
  data,
  n_iterations,
  n_burnin = 0,
  params = list(),
  alpha_SM = 1,
  B = 1000,
  L = 1,
  q = 0.5,
  kernel,
  print_progress = TRUE,
  user_seed = 1234
)
}
\arguments{
\item{data}{a matrix or an array If a matrix the algorithm for
univariate time series is used, where each row is a time series. If an array, the algorithm is run for multivariate time series. Each slice of the array is a matrix where the rows are the dimensions of the time series.}

\item{n_iterations}{number of MCMC iterations.}

\item{n_burnin}{number of iterations that must be excluded when computing the posterior estimate.}

\item{params}{a list of parameters:

If the time series is univariate the following must be specified:

\itemize{
\item \code{a},\code{b},\code{c} parameters of the integrated likelihood.
\item \code{phi} correlation parameter in the likelihood.
}

If the time series is multivariate the following must be specified:

\itemize{
\item \code{k_0}, \code{nu_0}, \code{S_0}, \code{m_0} parameters of the integrated likelihood.
\item \code{phi} correlation parameter in the likelihood.
}

If data are epidemic diffusions:

\itemize{
\item \code{M} number of Monte Carlo iterations when computing the likelihood of the epidemic diffusion.
\item \code{xi} recovery rate fixed constant for each population at each time.
\item \code{a0}, \code{b0} parameters for the computation of the integrated likelihood of the epidemic diffusions.
\item \code{I0_var} variance for the Metropolis-Hastings estimation of the proportion of infected at time 0.
\item \code{avg_blk} prior average number of change points for each order.
}}

\item{alpha_SM}{\eqn{\alpha} for the split-merge main algorithm.}

\item{B}{number of orders for the normalization constant.}

\item{L}{number of split-merge steps for the proposal step.}

\item{q}{probability of a split in the split-merge proposal and acceleration step.}

\item{kernel}{can be "ts" if data are time series or "epi" if data are epidemic diffusions.}

\item{print_progress}{If TRUE (default) print the progress bar.}

\item{user_seed}{seed for random distribution generation.}
}
\value{
A \code{ClustCpObj} class object containing

\itemize{
\item \code{$data} Vector or matrix containing the data.
\item \code{$n_iterations} Total number of MCMC iterations.
\item \code{$n_burnin} Number of burn-in iterations.
\item \code{$clust} A matrix where each row corresponds to the cluster assignment from each iteration.
\item \code{$orders} A multidimensional array where each slice is a matrix representing the latent order at each iteration.
\item \code{$time} Total computational time (in seconds).
\item \code{$entropy_MCMC} A \code{coda::mcmc} object containing the MCMC samples of the entropy.
\item \code{$lkl} A \code{coda::mcmc} object containing the log-likelihood evaluated at each iteration.
\item \code{$norm_vec} A vector containing the normalization constants computed at the beginning of the algorithm.
\item \code{$I0_MCMC} A \code{coda::mcmc} object containing the MCMC trace of the initial infection proportion \eqn{I_0}.
\item \code{$kernel_ts} TRUE if the kernel used corresponds to time series data.
\item \code{$kernel_epi} TRUE if the kernel used corresponds to epidemic diffusion data.
\item \code{$univariate_ts} TRUE if the data represent a univariate time series, FALSE if multivariate.
}
}
\description{
The \code{clust_cp} function cluster observations with common change points. Data can be time series or epidemic diffusions.
}
\examples{

\donttest{
## Univariate time series

data("stock_uni")

params_uni <- list(a = 1,
                   b = 1,
                   c = 1,
                   phi = 0.1)

out <- clust_cp(data = stock_uni[1:5,], n_iterations = 2000, n_burnin = 500,
                L = 1, q = 0.5, B = 1000, params = params_uni, kernel = "ts")

print(out)

## Multivariate time series

data("stock_multi")

params_multi <- list(m_0 = rep(0,2),
                     k_0 = 1,
                     nu_0 = 10,
                     S_0 = diag(1,2,2),
                     phi = 0.1)

out <- clust_cp(data = stock_multi[,,1:5], n_iterations = 2000, n_burnin = 500,
                L = 1, B = 1000, params = params_multi, kernel = "ts")

print(out)

## Epidemic diffusions

data("epi_synthetic_multi")

params_epi <- list(M = 100, xi = 1/8,
                   alpha_SM = 1,
                   a0 = 4,
                   b0 = 10,
                   I0_var = 0.1,
                   avg_blk = 2)

out <- clust_cp(epi_synthetic_multi, n_iterations = 2000, n_burnin = 500,
                L = 1, B = 1000, params = params_epi, kernel = "epi")

print(out)

}

}
\references{
Corradin, R., Danese, L., KhudaBukhsh, W. R., & Ongaro, A. (2026). Model-based clustering of time-dependent observations with common structural changes. \emph{Statistics and Computing}. \doi{10.1007/s11222-025-10756-x}
}
