test_that("calculate_mode works correctly", {
    set.seed(123)
    x <- rnorm(100, mean = 5, sd = 1)
    mode_val <- calculate_mode(x)

    expect_type(mode_val, "double")
    expect_length(mode_val, 1)
    expect_true(is.finite(mode_val))
    expect_true(mode_val > 3 && mode_val < 7)
})

test_that("calculate_mode handles edge cases", {
    expect_true(is.na(calculate_mode(numeric(0))))
    expect_equal(calculate_mode(5), 5)
    expect_true(is.finite(calculate_mode(c(1, 1, 1, 2, 2))))
})

test_that("safe_pct handles division correctly", {
    result <- safe_pct(c(10, 20, 30), c(100, 0, 200))

    expect_equal(result[1], 10)
    expect_true(is.na(result[2]))
    expect_equal(result[3], 15)
})

test_that("safe_rmse computes correctly", {
    actual <- c(1, 2, 3, 4, 5)
    predicted <- c(1, 2, 3, 4, 5)

    expect_equal(safe_rmse(actual, predicted), 0)

    predicted2 <- c(2, 3, 4, 5, 6)
    expect_equal(safe_rmse(actual, predicted2), 1)
})

test_that("safe_rmse handles NA values", {
    actual <- c(1, 2, NA, 4, 5)
    predicted <- c(1, 2, 3, 4, 5)

    result <- safe_rmse(actual, predicted)
    expect_true(is.finite(result))
})

test_that("safe_mae computes correctly", {
    actual <- c(1, 2, 3, 4, 5)
    predicted <- c(2, 3, 4, 5, 6)

    expect_equal(safe_mae(actual, predicted), 1)
})

test_that("robust_summary returns correct structure", {
    set.seed(123)
    x <- rnorm(50)

    result <- robust_summary(x)

    expect_type(result, "list")
    expect_named(result, c("mean", "sd", "median", "mad", "tmean", "ci"))
    expect_length(result$ci, 2)
})

test_that("robust_summary handles empty input", {
    result <- robust_summary(numeric(0))

    expect_true(is.na(result$mean))
    expect_true(is.na(result$sd))
})

test_that("evaluate_insample returns correct structure", {
    set.seed(123)
    actual <- log(runif(50, 100, 200))
    predicted <- actual + rnorm(50, 0, 0.1)

    result <- evaluate_insample(predicted, actual)

    expect_type(result, "list")
    expect_named(result, c("mae_log", "rmse_log", "mae_orig",
                           "rmse_orig", "mae_rel_range"))
    expect_true(all(sapply(result, is.finite)))
})

test_that("compute_r2 works correctly", {
    actual <- c(1, 2, 3, 4, 5)
    predicted <- c(1.1, 1.9, 3.1, 4.0, 4.9)

    r2 <- compute_r2(actual, predicted)

    expect_true(r2 > 0.9)
    expect_true(r2 <= 1)
})

test_that("compute_r2 handles perfect fit", {
    actual <- c(1, 2, 3, 4, 5)
    r2 <- compute_r2(actual, actual)

    expect_equal(r2, 1)
})

test_that("check_package works for available packages", {
    expect_true(check_package("stats", "testing"))
})

test_that("check_package fails for unavailable packages", {
    expect_error(
        check_package("nonexistent_package_xyz", "testing"),
        "not installed"
    )
})
