#' Convert formula-based model function to matrix interface
#'
#' @param fit_func Function accepting formula and data
#' @param predict_func Optional prediction function; if NULL, uses generic predict()
#' @param intercept Include intercept in formula (default TRUE)
#' @return List with fit and predict methods (class: model_adapter)
formula_to_matrix <- function(fit_func, predict_func = NULL, intercept = TRUE) {
  
  fit_wrapper <- function(X, y, weights = NULL, ...) {
    # Input validation
    if (!is.matrix(X) && !is.data.frame(X)) {
      stop("X must be a matrix or data frame")
    }
    if (length(y) != nrow(X)) {
      stop("Length of y must match number of rows in X")
    }
    if (!is.null(weights) && length(weights) != nrow(X)) {
      stop("Length of weights must match number of rows in X")
    }
    
    # Convert to data frame and ensure column names exist
    df <- as.data.frame(X)
    if (is.null(colnames(X))) {
      names(df) <- paste0("V", seq_len(ncol(X)))
    }
    
    # Create unique response name to avoid conflicts
    y_name <- ".response"
    while (y_name %in% names(df)) {
      y_name <- paste0(y_name, "_")
    }
    df[[y_name]] <- y
    
    # Handle weights if provided
    w_name <- NULL
    if (!is.null(weights)) {
      w_name <- ".weights"
      while (w_name %in% names(df)) {
        w_name <- paste0(w_name, "_")
      }
      df[[w_name]] <- weights
    }
    
    # Store factor levels and contrasts for prediction
    factor_info <- lapply(df, function(col) {
      if (is.factor(col)) {
        list(levels = levels(col), class = "factor")
      } else {
        list(levels = NULL, class = class(col)[1])
      }
    })
    
    # Build formula with proper quoting and intercept control
    predictor_cols <- setdiff(names(df), c(y_name, w_name))
    rhs <- paste(sprintf("`%s`", predictor_cols), collapse = " + ")
    if (!intercept) {
      rhs <- paste0("0 + ", rhs)
    }
    formula <- as.formula(paste(sprintf("`%s`", y_name), "~", rhs),
                          env = .GlobalEnv)  # Prevent environment bloat
    
    # Fit model, passing weights if present
    fit_args <- list(formula = formula, data = df, ...)
    if (!is.null(w_name)) fit_args$weights <- df[[w_name]]
    model <- do.call(fit_func, fit_args)
    
    # Store metadata
    attr(model, "._x_cols") <- predictor_cols
    attr(model, "._factor_info") <- factor_info[predictor_cols]
    attr(model, "._y_name") <- y_name
    
    return(model)
  }
  
  predict_wrapper <- function(model, newX, ...) {
    if (!is.matrix(newX) && !is.data.frame(newX)) {
      stop("newX must be a matrix or data frame")
    }
    
    newdf <- as.data.frame(newX)
    x_cols <- attr(model, "._x_cols")
    factor_info <- attr(model, "._factor_info")
    
    # Ensure column names match
    if (is.null(names(newdf))) {
      names(newdf) <- paste0("V", seq_len(ncol(newdf)))
    }
    if (!all(x_cols %in% names(newdf))) {
      stop("newX missing required columns: ",
           paste(setdiff(x_cols, names(newdf)), collapse = ", "))
    }
    
    # Reapply factor levels to ensure consistency
    for (col in x_cols) {
      info <- factor_info[[col]]
      if (!is.null(info$levels)) {
        newdf[[col]] <- factor(newdf[[col]], levels = info$levels)
      }
    }
    
    # Reorder columns to match training
    newdf <- newdf[, x_cols, drop = FALSE]
    
    # Predict
    if (is.null(predict_func)) {
      predict(model, newdata = newdf, ...)
    } else {
      predict_func(model, newdata = newdf, ...)
    }
  }
  
  structure(
    list(fit = fit_wrapper, predict = predict_wrapper),
    class = "model_adapter"
  )
}


#' Convert matrix-based model function to formula interface
#'
#' @param fit_func Function accepting x matrix and y vector
#' @param predict_func Optional prediction function; if NULL, uses generic predict()
#' @param drop_intercept Remove intercept column from model.matrix (default TRUE)
#' @return List with fit and predict methods (class: model_adapter)
matrix_to_formula <- function(fit_func, predict_func = NULL, drop_intercept = TRUE) {
  
  fit_wrapper <- function(formula, data, weights = NULL, ...) {
    # Input validation
    if (!inherits(formula, "formula")) {
      stop("formula must be a formula object")
    }
    if (!is.data.frame(data)) {
      stop("data must be a data frame")
    }
    
    # Extract response and design matrix, capturing factor levels
    mf <- model.frame(formula, data = data)
    if (nrow(mf) == 0) {
      stop("model.frame produced empty result")
    }
    
    y <- model.response(mf)
    terms_obj <- terms(mf)
    xlev <- .getXlevels(terms_obj, mf)  # Capture factor levels
    X <- model.matrix(formula, data = mf)
    
    # Handle intercept: most matrix-based learners don't want it
    intercept_removed <- FALSE
    if (drop_intercept && "(Intercept)" %in% colnames(X)) {
      X <- X[, colnames(X) != "(Intercept)", drop = FALSE]
      intercept_removed <- TRUE
    }
    
    # Fit model, passing weights if provided
    # Use lowercase 'x' to match common conventions (glmnet, xgboost, etc.)
    fit_args <- list(x = X, y = y, ...)
    if (!is.null(weights)) {
      # Check if fit_func accepts weights parameter
      if ("weights" %in% names(formals(fit_func))) {
        fit_args$weights <- weights
      } else {
        warning("fit_func does not accept 'weights' parameter; ignoring")
      }
    }
    model <- do.call(fit_func, fit_args)
    
    # Store metadata for prediction
    attr(model, "._formula") <- formula
    attr(model, "._terms") <- terms_obj
    attr(model, "._xlev") <- xlev
    attr(model, "._x_cols") <- colnames(X)
    attr(model, "._intercept_removed") <- intercept_removed
    
    return(model)
  }
  
  predict_wrapper <- function(model, newdata, ...) {
    if (!is.data.frame(newdata)) {
      stop("newdata must be a data frame")
    }
    
    # Reconstruct design matrix using stored terms and factor levels
    terms_obj <- attr(model, "._terms")
    xlev <- attr(model, "._xlev")
    
    # Use xlev to ensure consistent factor handling
    mf_new <- model.frame(delete.response(terms_obj), newdata, 
                          xlev = xlev, na.action = na.pass)
    newX <- model.matrix(delete.response(terms_obj), data = mf_new)
    
    # Apply same intercept handling as training
    if (attr(model, "._intercept_removed") && "(Intercept)" %in% colnames(newX)) {
      newX <- newX[, colnames(newX) != "(Intercept)", drop = FALSE]
    }
    
    # Ensure column alignment with training
    x_cols <- attr(model, "._x_cols")
    if (!all(x_cols %in% colnames(newX))) {
      missing <- setdiff(x_cols, colnames(newX))
      stop("Prediction data missing required columns: ", paste(missing, collapse = ", "))
    }
    
    # Reorder columns to match training exactly
    newX <- newX[, x_cols, drop = FALSE]
    
    # Predict
    if (is.null(predict_func)) {
      predict(model, newX, ...)
    } else {
      predict_func(model, newX, ...)
    }
  }
  
  structure(
    list(fit = fit_wrapper, predict = predict_wrapper),
    class = "model_adapter"
  )
}


#' Print Method for Model Adapters
#'
#' Prints a short summary of a \code{model_adapter} object, including the
#' signatures of its \code{fit} and \code{predict} functions.
#'
#' @param x An object of class \code{model_adapter}.
#' @param ... Further arguments passed to or from other methods (ignored).
#'
#' @return Invisibly returns \code{x}.
#'
#' @method print model_adapter
#' @export
#'
#' @examples
#' adapter <- list(
#'   fit = function(data, y) NULL,
#'   predict = function(newdata) numeric(0)
#' )
#' class(adapter) <- "model_adapter"
#' print(adapter)
print.model_adapter <- function(x, ...) {
  cat("Model Interface Adapter\n")
  cat(
    "  fit:     function(",
    paste(names(formals(x$fit)), collapse = ", "),
    ")\n",
    sep = ""
  )
  cat(
    "  predict: function(",
    paste(names(formals(x$predict)), collapse = ", "),
    ")\n",
    sep = ""
  )
  invisible(x)
}



