#'@title Time Series Smooth
#'@description Remove or reduce randomness (noise) using a robust smoothing
#' strategy that first mitigates outliers and then smooths residual variation.
#'@return A `ts_fil_smooth` object.
#'@examples
#'# Robust smoothing with iterative outlier mitigation
#' # Load package and example data
#' library(daltoolbox)
#' data(tsd)
#' tsd$y[9] <- 2 * tsd$y[9]  # inject an outlier
#'
#' # Fit smoother and transform to reduce spikes/noise
#' filter <- ts_fil_smooth()
#' filter <- fit(filter, tsd$y)
#' y <- transform(filter, tsd$y)
#'
#' # Compare original vs smoothed series
#' plot_ts_pred(y = tsd$y, yadj = y)
#'@importFrom daltoolbox dal_transform
#'@importFrom daltoolbox fit
#'@importFrom daltoolbox transform
#'@export
ts_fil_smooth <- function() {
  obj <- dal_transform()
  class(obj) <- append("ts_fil_smooth", class(obj))
  return(obj)
}

#'@importFrom stats na.omit
#'@importFrom graphics boxplot
#'@importFrom daltoolbox transform
#'@exportS3Method transform ts_fil_smooth
transform.ts_fil_smooth <- function(obj, data, ...) {
  progressive_smoothing <- function(serie) {
    serie <- stats::na.omit(serie)
    repeat {
      n <- length(serie)
      diff <- serie[2:n] - serie[1:(n-1)]

      names(diff) <- 1:length(diff)
      bp <- graphics::boxplot(diff, plot = FALSE)
      j <- as.integer(names(bp$out))

      rj <- j[(j > 1) & (j < length(serie))]
      serie[rj] <- (serie[rj-1]+serie[rj+1])/2

      diff <- serie[2:n] - serie[1:(n-1)]
      bpn <- graphics::boxplot(diff, plot = FALSE)

      if ((length(bpn$out) == 0) || (length(bp$out) == length(bpn$out))) {
        break
      }
    }
    return(serie)
  }
  xd <- progressive_smoothing(data)
  return(xd)
}

