% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h_response_subgroups.R
\name{h_response_subgroups}
\alias{h_response_subgroups}
\alias{h_proportion_df}
\alias{h_proportion_subgroups_df}
\alias{h_odds_ratio_df}
\alias{h_odds_ratio_subgroups_df}
\title{Helper functions for tabulating binary response by subgroup}
\usage{
h_proportion_df(rsp, arm)

h_proportion_subgroups_df(
  variables,
  data,
  groups_lists = list(),
  label_all = "All Patients"
)

h_odds_ratio_df(rsp, arm, strata_data = NULL, conf_level = 0.95, method = NULL)

h_odds_ratio_subgroups_df(
  variables,
  data,
  groups_lists = list(),
  conf_level = 0.95,
  method = NULL,
  label_all = "All Patients"
)
}
\arguments{
\item{rsp}{(\code{logical})\cr vector indicating whether each subject is a responder or not.}

\item{arm}{(\code{factor})\cr the treatment group variable.}

\item{variables}{(named \code{list} of \code{string})\cr list of additional analysis variables.}

\item{data}{(\code{data.frame})\cr the dataset containing the variables to summarize.}

\item{groups_lists}{(named \code{list} of \code{list})\cr optionally contains for each \code{subgroups} variable a
list, which specifies the new group levels via the names and the
levels that belong to it in the character vectors that are elements of the list.}

\item{label_all}{(\code{string})\cr label for the total population analysis.}

\item{strata_data}{(\code{factor}, \code{data.frame}, or \code{NULL})\cr required if stratified analysis is performed.}

\item{conf_level}{(\code{proportion})\cr confidence level of the interval.}

\item{method}{(\code{string} or \code{NULL})\cr specifies the test used to calculate the p-value for the difference between
two proportions. For options, see \code{\link[=test_proportion_diff]{test_proportion_diff()}}. Default is \code{NULL} so no test is performed.}
}
\value{
\itemize{
\item \code{h_proportion_df()} returns a \code{data.frame} with columns \code{arm}, \code{n}, \code{n_rsp}, and \code{prop}.
}

\itemize{
\item \code{h_proportion_subgroups_df()} returns a \code{data.frame} with columns \code{arm}, \code{n}, \code{n_rsp}, \code{prop}, \code{subgroup},
\code{var}, \code{var_label}, and \code{row_type}.
}

\itemize{
\item \code{h_odds_ratio_df()} returns a \code{data.frame} with columns \code{arm}, \code{n_tot}, \code{or}, \code{lcl}, \code{ucl}, \code{conf_level}, and
optionally \code{pval} and \code{pval_label}.
}

\itemize{
\item \code{h_odds_ratio_subgroups_df()} returns a \code{data.frame} with columns \code{arm}, \code{n_tot}, \code{or}, \code{lcl}, \code{ucl},
\code{conf_level}, \code{subgroup}, \code{var}, \code{var_label}, and \code{row_type}.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Helper functions that tabulate in a data frame statistics such as response rate
and odds ratio for population subgroups.
}
\details{
Main functionality is to prepare data for use in a layout-creating function.
}
\section{Functions}{
\itemize{
\item \code{h_proportion_df()}: Helper to prepare a data frame of binary responses by arm.

\item \code{h_proportion_subgroups_df()}: Summarizes proportion of binary responses by arm and across subgroups
in a data frame. \code{variables} corresponds to the names of variables found in \code{data}, passed as a named list and
requires elements \code{rsp}, \code{arm} and optionally \code{subgroups}. \code{groups_lists} optionally specifies
groupings for \code{subgroups} variables.

\item \code{h_odds_ratio_df()}: Helper to prepare a data frame with estimates of
the odds ratio between a treatment and a control arm.

\item \code{h_odds_ratio_subgroups_df()}: Summarizes estimates of the odds ratio between a treatment and a control
arm across subgroups in a data frame. \code{variables} corresponds to the names of variables found in
\code{data}, passed as a named list and requires elements \code{rsp}, \code{arm} and optionally \code{subgroups}
and \code{strata}. \code{groups_lists} optionally specifies groupings for \code{subgroups} variables.

}}
\examples{
library(dplyr)
library(forcats)

adrs <- tern_ex_adrs
adrs_labels <- formatters::var_labels(adrs)

adrs_f <- adrs \%>\%
  filter(PARAMCD == "BESRSPI") \%>\%
  filter(ARM \%in\% c("A: Drug X", "B: Placebo")) \%>\%
  droplevels() \%>\%
  mutate(
    # Reorder levels of factor to make the placebo group the reference arm.
    ARM = fct_relevel(ARM, "B: Placebo"),
    rsp = AVALC == "CR"
  )
formatters::var_labels(adrs_f) <- c(adrs_labels, "Response")

h_proportion_df(
  c(TRUE, FALSE, FALSE),
  arm = factor(c("A", "A", "B"), levels = c("A", "B"))
)

h_proportion_subgroups_df(
  variables = list(rsp = "rsp", arm = "ARM", subgroups = c("SEX", "BMRKR2")),
  data = adrs_f
)

# Define groupings for BMRKR2 levels.
h_proportion_subgroups_df(
  variables = list(rsp = "rsp", arm = "ARM", subgroups = c("SEX", "BMRKR2")),
  data = adrs_f,
  groups_lists = list(
    BMRKR2 = list(
      "low" = "LOW",
      "low/medium" = c("LOW", "MEDIUM"),
      "low/medium/high" = c("LOW", "MEDIUM", "HIGH")
    )
  )
)

# Unstratatified analysis.
h_odds_ratio_df(
  c(TRUE, FALSE, FALSE, TRUE),
  arm = factor(c("A", "A", "B", "B"), levels = c("A", "B"))
)

# Include p-value.
h_odds_ratio_df(adrs_f$rsp, adrs_f$ARM, method = "chisq")

# Stratatified analysis.
h_odds_ratio_df(
  rsp = adrs_f$rsp,
  arm = adrs_f$ARM,
  strata_data = adrs_f[, c("STRATA1", "STRATA2")],
  method = "cmh"
)

# Unstratified analysis.
h_odds_ratio_subgroups_df(
  variables = list(rsp = "rsp", arm = "ARM", subgroups = c("SEX", "BMRKR2")),
  data = adrs_f
)

# Stratified analysis.
h_odds_ratio_subgroups_df(
  variables = list(
    rsp = "rsp",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2"),
    strata = c("STRATA1", "STRATA2")
  ),
  data = adrs_f
)

# Define groupings of BMRKR2 levels.
h_odds_ratio_subgroups_df(
  variables = list(
    rsp = "rsp",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2")
  ),
  data = adrs_f,
  groups_lists = list(
    BMRKR2 = list(
      "low" = "LOW",
      "low/medium" = c("LOW", "MEDIUM"),
      "low/medium/high" = c("LOW", "MEDIUM", "HIGH")
    )
  )
)

}
