#' Calculate the Average of a Numeric Vector
#'
#' This function calculates the arithmetic mean of the elements in a numeric vector.
#' The implementation uses a `for` loop to accumulate the sum of the elements
#' and then divides the total by the number of elements in the vector.
#'
#' @param vector A numeric vector whose elements are to be averaged.
#'
#' @return A number representing the average of the elements in the vector.
#'
#' @keywords internal
#' @noRd
.mean <- function(vector) {
  n <- length(vector)
  count <- 0
  for (i in 1:n) {
    count <- count + vector[i]
  }
  count / n
}

#' Calculate the Population Variance of a Numeric Vector
#'
#' This function calculates the population variance of a set of numerical data.
#' It uses a `for` loop to accumulate the squared deviations from the mean,
#' and then divides the sum by the total number of elements.
#'
#' This implementation relies on the function `.mean()`, which must be defined beforehand.
#'
#' @param vector A numeric vector for which the population variance is to be calculated.
#'
#' @return A number representing the population variance of the elements in the vector.
#'
#' @seealso \code{\link{.mean}}
#'
#' @keywords internal
#' @noRd
variance <- function(vector) {
  n <- length(vector)
  mu <- .mean(vector)
  count <- 0
  for (i in 1:n) {
    count <- count + (vector[i] - mu)^2
  }
  count / n
}

#' Calculate the Population Standard Deviation of a Numeric Vector
#'
#' This function returns the population standard deviation of a set of numerical data.
#' Internally, it uses the function `variance()` to calculate the population variance,
#' and then applies the `sqrt()` function to obtain the square root.
#'
#' This function depends on `variance()` and, therefore, indirectly also on `.mean()`.
#'
#' @param vector A numeric vector for which the population standard deviation is to be calculated.
#'
#' @return A number representing the population standard deviation of the vector.
#'
#' @seealso \code{\link{variance}}, \code{\link{.mean}}
#'
#' @keywords internal
#' @noRd
standard.deviation <- function(vector) {
  sqrt(variance(vector))
}

#' Center a Matrix by Columns by Subtracting the Mean of Each Column
#'
#' This function transforms a numeric matrix by subtracting the mean of each column
#' from every element in that column. The result is a centered matrix, where each column
#' has a mean equal to zero.
#'
#' Internally, it converts the input object to a matrix (if it isn't already) and uses the
#' function `.mean()` to compute the mean of each column.
#'
#' @param original.matrix A numeric matrix or data frame whose columns are to be centered.
#'
#' @return A numeric matrix centered by columns.
#'
#' @seealso \code{\link{.mean}}
#'
#' @keywords internal
#' @noRd
center <- function(original.matrix) {
  # Convert the input to a matrix to ensure consistent numeric operations.
  matrix <- as.matrix(original.matrix)
  m <- ncol(matrix)
  for (i in 1:m) {
    col <- matrix[, i] # Extract the i-th column
    mu <- .mean(col)
    matrix[, i] <- col - mu # Subtract the mean to center the column
  }
  matrix
}

#' Center and Scale a Numeric Matrix by Columns
#'
#' This function standardizes a numeric matrix by transforming each column
#' to have a mean of 0 and a standard deviation of 1.
#' Internally, it uses the functions `.mean()` and `standard.deviation()`.
#'
#' In the case where a column has a standard deviation of 0
#' (i.e., all its values are the same), the column is replaced with zeros.
#'
#' @param original.matrix A numeric matrix or data frame to be standardized.
#'
#' @return A numeric matrix with centered (mean zero) and scaled (standard deviation one) columns.
#'
#' @seealso \code{\link{.mean}}, \code{\link{standard.deviation}}
#'
#' @noRd
#'
#' @keywords internal
center.and.reduce <- function(original.matrix) {
  matrix <- as.matrix(original.matrix)
  m <- ncol(matrix)
  for (i in 1:m) {
    col <- matrix[, i]
    mu <- .mean(col)
    sigma <- standard.deviation(col)
    if (sigma == 0) {
      matrix[, i] <- 0
    } else {
      matrix[, i] <- (col - mu) / sigma # Standard Z-score transformation
    }
  }
  matrix
}
