#' Fit Bayesian generalised (non-)linear multilevel compositional model
#' via full Bayesian inference
#'
#' Fit a \code{brm} model with multilevel ILR coordinates
#'
#' @param complr A \code{\link{complr}} object containing data of composition,
#' ILR coordinates, and other variables used in the model.
#' @param formula A object of class \code{formula}, \code{brmsformula}:
#' A symbolic description of the model to be fitted.
#' Details of the model specification can be found in \code{\link[brms:brmsformula]{brmsformula}}.
#' @param ... Further arguments passed to \code{\link[brms:brm]{brm}}.
#'
#' @return A \code{\link{brmcoda}} with two elements
#'   \item{\code{complr}}{ An object of class \code{complr} used in the \code{brm} model. }
#'   \item{\code{model}}{ An object of class \code{brmsfit}, which contains the posterior draws
#'   along with many other useful information about the model.}
#' @importFrom brms brm
#'
#' @examples
#' \donttest{
#' if(requireNamespace("cmdstanr")){
#'   x1 <- complr(data = mcompd, sbp = sbp,
#'                  parts = c("TST", "WAKE", "MVPA", "LPA", "SB"), idvar = "ID")
#'
#'   # inspect variables before passing to brmcoda
#'   get_variables(x1)
#'
#'   ## model with compositional predictor at between and within-person levels
#'   m1 <- brmcoda(complr = x1,
#'                 formula = Stress ~ bz1_1 + bz2_1 + bz3_1 + bz4_1 +
#'                                    wz1_1 + wz2_1 + wz3_1 + wz4_1 + (1 | ID),
#'                 chain = 1, iter = 500,
#'                 backend = "cmdstanr")
#'
#'   ## model with compositional outcome
#'   m2 <- brmcoda(complr = x1,
#'                 formula = mvbind(z1_1, z2_1, z3_1, z4_1) ~ Stress + Female + (1 | ID),
#'                 chain = 1, iter = 500,
#'                 backend = "cmdstanr")
#'
#'   ## model with compositional predictor and outcome
#'   x2 <- complr(data = mcompd,
#'                 parts = list(c("TST", "WAKE"), c("MVPA", "LPA", "SB")),
#'                 total = list(c(480), c(960)),
#'                 idvar = "ID",
#'                 transform = "ilr")
#'
#'   m3 <- brmcoda(complr = x2,
#'                 formula = mvbind(z1_2, z2_2) ~ z1_1 + Female + (1 | ID),
#'                 chain = 1, iter = 500,
#'                 backend = "cmdstanr")
#'   }}
#' @export
brmcoda <- function (complr, formula, ...) {
  if (isTRUE(missing(complr))) {
    stop(
      paste(
        "'complr' is a required argument and cannot be missing;",
        "  it should be an object of class complr.",
        "  See ?multilevelcoda::complr for details.",
        sep = "\n"
      )
    )
  }
  if (isFALSE(inherits(complr, "complr"))) {
    stop(
      paste(
        "complr must be an object of class complr.",
        "  See ?multilevelcoda::complr for details.",
        sep = "\n"
      )
    )
  }
  
  ## if formula contains ilr, print a message
  fterms <- all.vars(formula)
  zterms <- get_variables(complr)
  
  zterms <- grep("z", unlist(zterms), value = TRUE)
  if (any(grepl("ilr", fterms))) {
    warning(
      sprintf(
        "We recently updated naming convention for the ilrs. Please use the 'z' terms generated by complr(): %s",
        paste(zterms, collapse = ", ")
      )
    )
  }
  
  # fit the model
  m <- brm(formula, data = complr$dataout, ...)
  structure(list(complr = complr, model = m), class = "brmcoda")
}
