% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/therm_suit_bounds.R
\name{therm_suit_bounds}
\alias{therm_suit_bounds}
\title{Determine Thermal Boundaries for Optimal Performance Level}
\usage{
therm_suit_bounds(
  preds_tbl = NULL,
  model_name = NULL,
  suitability_threshold = NULL
)
}
\arguments{
\item{preds_tbl}{a \code{tibble} object as produced by \code{\link[=predict_curves]{predict_curves()}}.}

\item{model_name}{character. Name of one or several of the TPC models fitted
first in \code{fit_devmodels()} and predicted next in \code{predict_curves()}.
If using \code{model_name = "all"} all models contained in \code{preds_tbl} will be used.
Please, note that some models (most typically, \code{briere}, \code{mod_poly}, \code{wang} and
\code{ratkowsky}) may calculate unrealistic thermal boundaries. We
recommend to double-check it and compare among several models.}

\item{suitability_threshold}{A numeric value from 50 to 100 representing
the quantile of the curve that provides the user-defined optimal performance.
For instance, setting \code{suitability_threshold} to 80 identifies the top 20\%
(or quantile 80) of the maximum values of the development rate predicted by
the chosen TPC model. If \code{suitability_threshold} equals 100, the function
returns the optimum temperature for development rate.}
}
\value{
A tibble with six columns:
\itemize{
\item \code{model_name}: A string indicating the selected TPC model used for projections.
\item \code{suitability}: A string indicating the suitability threshold in percentage
(see \code{suitability_threshold}).
\item \code{tval_left}: A number representing the lower thermal boundary delimiting
the suitable region of the TPC.
\item \code{tval_right}: A number representing the upper thermal boundary delimiting
the suitable region of the TPC.
\item \code{pred_suit}: A number corresponding to the predicted development rate value
determining the chosen quantile threshold of the maximum rate
(i.e., suitability percentage of maximum rate).
\item \code{iter}: A string determining the TPC identity from the bootstrapping
procedure in \code{\link[=predict_curves]{predict_curves()}} function, or \code{estimate} when it represents
the estimated TPC fitted in \code{\link[=fit_devmodels]{fit_devmodels()}}.
}
}
\description{
Calculate thermal boundaries that define the suitable region of a
Thermal Performance Curve (TPC) corresponding to a user-defined optimal performance level.
}
\examples{
\dontshow{if (interactive()) withAutoprint(\{ # examplesIf}
data("aphid")

fitted_tpcs <- fit_devmodels(temp = aphid$temperature,
                             dev_rate = aphid$rate_value,
                             model_name = "all")

plot_devmodels(temp = aphid$temperature,
               dev_rate = aphid$rate_value,
               fitted_parameters = fitted_tpcs,
               species = "Brachycaudus schwartzi",
               life_stage = "Nymphs")

boot_tpcs <- predict_curves(temp = aphid$temperature,
                            dev_rate = aphid$rate_value,
                            fitted_parameters = fitted_tpcs,
                            model_name_2boot = c("lactin2", "briere2", "beta"),
                            propagate_uncertainty = TRUE,
                            n_boots_samples = 10)

print(boot_tpcs)


plot_uncertainties(temp = aphid$temperature,
                   dev_rate = aphid$rate_value,
                   bootstrap_tpcs = boot_tpcs,
                   species = "Brachycaudus schwartzi",
                   life_stage = "Nymphs")


boundaries <- therm_suit_bounds(preds_tbl = boot_tpcs,
                                model_name = "lactin2",
                                suitability_threshold = 80)
head(boundaries)
\dontshow{\}) # examplesIf}
}
\references{
Angilletta, M.J., (2006). Estimating and comparing thermal performance curves.
\if{html}{\out{<i>}}J. Therm. Biol.\if{html}{\out{</i>}} 31: 541-545. (for model selection in TPC framework)

Padfield, D., O'Sullivan, H. and Pawar, S. (2021). \if{html}{\out{<i>}}rTPC\if{html}{\out{</i>}} and \if{html}{\out{<i>}}nls.multstart\if{html}{\out{</i>}}:
A new pipeline to fit thermal performance curves in \code{R}. \if{html}{\out{<i>}}Methods Ecol Evol\if{html}{\out{</i>}}. 12: 1138-1143.

Rebaudo, F., Struelens, Q. and Dangles, O. (2018). Modelling temperature-dependent
development rate and phenology in arthropods: The \code{devRate} package for \code{R}.
\if{html}{\out{<i>}}Methods Ecol Evol\if{html}{\out{</i>}}. 9: 1144-1150.

Satar, S. and Yokomi, R. (2002). Effect of temperature and host on development
of \if{html}{\out{<i>}}Brachycaudus schwartzi\if{html}{\out{</i>}} (Homoptera: Aphididae).
\if{html}{\out{<i>}}Ann. Entomol. Soc. Am.\if{html}{\out{</i>}} 95: 597-602.
}
\seealso{
\code{browseVignettes("rTPC")} for model names, start values searching workflows, and
bootstrapping procedures using both \code{\link[rTPC:get_start_vals]{rTPC::get_start_vals()}} and \code{\link[nls.multstart:nls_multstart]{nls.multstart::nls_multstart()}}

\code{\link[=fit_devmodels]{fit_devmodels()}} for fitting Thermal Performance Curves to development rate data,
which is in turn based on \code{\link[nls.multstart:nls_multstart]{nls.multstart::nls_multstart()}}.
\code{\link[=predict_curves]{predict_curves()}} for bootstrapping procedure based on the above-mentioned \code{rTPC} vignettes.
}
