#' @title A Macro Pre-processor for 'R' Programs
#'
#' @description The \strong{macro} package contains a function to
#' preprocess R scripts, and output a resolved code file.  Pre-processor
#' commands are implemented as special comments.
#'
#' @details
#' The \strong{macro} package attempts to create a macro language for R that is
#' similar to the SAS® macro language.
#'
#' There is only one function in the package:
#' \code{\link{msource}}.  This function acts as a substitute for the Base R
#' \code{source} function.  The difference is that \code{msource} first runs
#' a pre-processor to resolve macro statements. The resolved macro statements
#' are emitted to a separate file.  The \code{msource} function then sources that
#' file normally.
#'
#' By default, the pre-processor output is written to a temporary location,
#' and deleted when the operation is complete.  If desired, you may also
#' supply a path to save this file to a location of your choosing.
#'
#' @section Macro Commands:
#' Macro commands are implemented as a special form of R comment. The special
#' comments begin with "#%".  For example, a macro assignment is written as
#' \code{#%let a <- 1}.
#'
#' Here are the macro features supported by the system:
#' \itemize{
#'   \item \strong{Macro Comments}: A comment just for macro statements.
#'   \item \strong{Macro Variables}: Initialize macro variables to be used as text replacement tokens.
#'   \item \strong{Macro Conditionals}: Emit code conditionally in pre-processing.
#'   \item \strong{Macro Include}: Insert code from external files into your program.
#'   \item \strong{Built-In Macro Functions}: A small number of essential built-in macro functions.
#'   \item \strong{Macro Do Loops}: Emits a block of code repeatedly.
#'   \item \strong{User-Defined Macro Functions}: Custom macro functions to
#'   enhance reuse and reduce redundancy.
#'   \item \strong{Macro Line Continuation}: A line continuation operator that
#'   allows you to write macro commands that span more than one line.
#' }
#' The above features give you a simple yet flexible way to perform meta-programming.
#' In this way, the \strong{macro} package can be useful in several situations,
#' notably for code generation.
#'
#' See the \code{\link{msource}} function documentation for additional details.
#' @name macro
#' @aliases macro-package
#' @keywords internal
"_PACKAGE"


#' @title Addin Function to Run msource()
#' @description
#' This function is exposed to the addin menu to run \code{msource()}
#' interactively. The function will run either the
#' currently selected code, or the currently active program.
#' The function sets the "envir" parameter
#' to the global environment and the "clear" parameter to FALSE
#' to enhance the user experience.  The function
#' takes no parameters and is only used by the addin menu.
#' @return The results of \code{msource}, invisibly.
#' @examples
#' \donttest{
#' # Called from addin menu
#' runMSource()
#' }
#' @export
runMSource <- function() {

  if (rstudioapi::isAvailable()) {

    # Load macro package if not already loaded
    if (!"macro" %in% .packages()) {
      require(macro, quietly = TRUE, warn.conflicts = FALSE)
    }

    # Save current file if needed
    autoSave()

    # Run msource()
    res <- tryCatch({msource(envir = parent.frame(), echo = TRUE, clear = FALSE)},
                    warning = function(cond) {
                      # print("here!")
                      return(cond)
                      },
                    error = function(cond) {return(cond)})

    # Deal with error messages
    ret <- processMessages(res)

  } else {
    ret <- NULL
  }

  invisible(ret)
}



#' @title Addin Function to Run msource() in Debug Mode
#' @description
#' This function is exposed to the addin menu to run \code{msource()}
#' interactively in debug mode. The function will run either the
#' currently selected code, or the currently active program.
#' On the call to \code{msource}, it sets the "debug" and "symbolgen"
#' parameters to TRUE.  The function also sets the "envir" parameter
#' to the global environment and the "clear" parameter to FALSE
#' to enhance the user experience.  The function
#' takes no parameters and is only used by the addin menu.
#' @return The results of \code{msource}, invisibly.
#' @examples
#' \donttest{
#' # Called from addin menu
#' runMSourceDebug()
#' }
#' @export
runMSourceDebug <- function() {

  if (rstudioapi::isAvailable()) {

    # Load macro package if not already loaded
    if (!"macro" %in% .packages()) {
      require(macro, quietly = TRUE, warn.conflicts = FALSE)
    }

    # Save current file if needed
    autoSave()

    # Run msource() in debug mode
    res <- tryCatch({msource(debug = TRUE, symbolgen = TRUE,
                             envir = parent.frame(), clear = FALSE)},
                    warning = function(cond) {
                      # print("here!")
                      return(cond)
                      },
                    error = function(cond) { return(cond) })

    # Deal with error messages
    ret <- processMessages(res)

  } else {

    ret <- NULL
  }

  invisible(ret)

}

# This function is necessary because RStudio wants to pop up
# a dialog every time a real error occurs.  The only way to get rid
# of the popup is to trap the error and just send error text to the console.
# Warnings are a whole different problem.
#' @noRd
processMessages <- function(res) {

  ret <- res

  if ("warning" %in% class(res)) {

    # See comment below
    cat(warnings())

  }

  if ("error" %in% class(res)) {

    if (length(res$call) > 1) {
      cl <- res$call[1]
    } else {
      cl <- res$call
    }

    msg <- paste0("Error in ", cl, ": ", res$message, "\n", collapse = "\n")

    # NOTE to CRAN: The cat() is needed because these functions are called
    # from an RStudio addin.  When called from an RStudio addin, RStudio
    # has this very annoying behavior of popping up a dialog with the
    # error or warning message.  I just want the error or warning message
    # sent to the console like normal, and don't want the pop-up dialog.
    # The only work-around I can find is to trap the error/warning and cat()
    # it to the console so the user gets the error/warning message,
    # but does not have to get a pop-up dialog.  Please allow the cat()
    # in this case. It is only to help the user.
    cat(msg)
  }

  return(ret)

}

# Deal with autosave choices. Behavior is to automatically save the opened
# document if the user is executing the entire document.  If there are unsaved
# changes, those changes have to be saved because msource() looks at the file
# on disk.  On the other hand, if the user is only selecting a few lines of
# code to execute, there is no need to save the entire document.  Desired behavior
# is similar to RStudio source button.
#' @noRd
autoSave <- function() {

  # Basic requirement
  if (length(find.package('rstudioapi', quiet=TRUE)) > 0) {

    # See if user is selecting anything
    sel <- get_selection()

    # If user is selecting something, don't save the document
    if (length(sel) == 1 & sel[1] == "") {


      if (is.null(options()[["macro.autosave"]]) == FALSE) {

        opt <- options("macro.autosave")

        # Only save if TRUE
        if (all(opt[[1]] == TRUE)) {
          id <- rstudioapi::documentId(FALSE)

          rstudioapi::documentSave(id)
        }

      } else {  # Default is to save document
        id <- rstudioapi::documentId(FALSE)

        rstudioapi::documentSave(id)
      }



    }
  }
}

