\name{savePlotly}
\alias{savePlotly}
\title{Save the last Plotly chart to a standalone HTML file}

\description{
Save a Plotly/htmlwidget chart to an HTML file. If no object is supplied,
\code{savePlotly()} uses the most recent Plotly chart generated by
\pkg{lessR} functions (cached in \code{options("lessR.last_plotly")}).
A sensible filename is auto-generated from chart metadata (e.g., \dQuote{pie},
variable names) or from the chart title if metadata are unavailable.
Optionally opens the saved file in your default web browser.
}

\usage{
savePlotly(obj = NULL, file = NULL, open = TRUE,
           selfcontained = TRUE, libdir = NULL)
}

\arguments{
  \item{obj}{A Plotly/htmlwidget object. If \code{NULL} (default), the function
    attempts to retrieve the last chart from \code{options("lessR.last_plotly")}.}
  \item{file}{Output path for the HTML file. If \code{NULL}, a filename is
    constructed in the current working directory using the chart kind and
    variables, e.g., \code{plotly_pie_Dept.html} or
    \code{plotly_pie_DeptGender.html}.}
  \item{open}{Logical. If \code{TRUE} (default), open the saved file in the
    system's default web browser.}
  \item{selfcontained}{Logical. Passed to \code{htmlwidgets::saveWidget()}.
    If \code{TRUE}, bundle all dependencies into a single HTML file.}
  \item{libdir}{Optional directory to copy widget dependencies when
    \code{selfcontained = FALSE}. Passed to \code{htmlwidgets::saveWidget()}.}
}

\details{
\code{savePlotly()} is designed for seamless export of interactive outputs
created by \pkg{lessR} Plotly functions When the
\code{obj} parameter is \code{NULL}, the function retrieves the most-recent
Plotly widget stored by lessR in \code{options("lessR.last_plotly")}.

If \code{file} is not provided, a filename is automatically derived from
attributes embedded in the widget (when available), specifically:
\itemize{
  \item \code{lessR_kind} (e.g., \dQuote{pie})
  \item \code{lessR_xname} (primary variable name)
  \item \code{lessR_byname} (grouping variable name, if any)
}
If those attributes are missing, the function falls back to parsing the plain
chart title (e.g., \dQuote{Gender by Dept}). Characters unsuitable for filenames
are removed.

The function prints the absolute path of the saved file to the console and
(invisibly) returns that path. Setting \code{open = TRUE} will launch the file
in your default browser after saving.
}

\value{
(Invisibly) returns the normalized file path of the saved HTML file (a character
scalar).
}

\section{Side Effects}{
\itemize{
  \item Prints a confirmation message with the save location.
  \item Optionally opens the saved HTML file in the default browser when
        \code{open = TRUE}.
}
}

\seealso{
\code{\link[htmlwidgets]{saveWidget}},
}

\examples{
## Not run: 

d <- Read("Employee")

if (interactive()) {
  # Basic usage with the most recent lessR Plotly chart
  PieChart(Dept, data=d)
  savePlotly()  # -> e.g., "plotly_pie_Dept.html"

  # Grouped example; filename reflects both variables
  PieChart(Dept, by=Gender, data=d)
  savePlotly()  # -> e.g., "plotly_pie_DeptGender.html"

  # Explicit object and custom path
  plt <- PieChart(Dept, data=d)
  savePlotly(plt, file=file.path(tempdir(), "my_dept_pie.html"), open=FALSE)

  # Non-selfcontained (assets in a side folder)
  savePlotly(plt, file="dept_pie.html", selfcontained=FALSE,
             libdir="dept_pie_libs")
  ## End(Not run)
  }
}

\note{
Requires the \pkg{htmlwidgets} package to be installed.
If no recent Plotly chart is available and \code{obj} is \code{NULL},
the function will error and prompt you to pass a chart or render one first.
}
\author{lessR Development Team}
\keyword{hplot}
