% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layers-normalization.R
\name{layer_batch_normalization}
\alias{layer_batch_normalization}
\title{Layer that normalizes its inputs}
\usage{
layer_batch_normalization(
  object,
  axis = -1L,
  momentum = 0.99,
  epsilon = 0.001,
  center = TRUE,
  scale = TRUE,
  beta_initializer = "zeros",
  gamma_initializer = "ones",
  moving_mean_initializer = "zeros",
  moving_variance_initializer = "ones",
  beta_regularizer = NULL,
  gamma_regularizer = NULL,
  beta_constraint = NULL,
  gamma_constraint = NULL,
  synchronized = FALSE,
  ...
)
}
\arguments{
\item{object}{Layer or model object}

\item{axis}{Integer, the axis that should be normalized (typically the features
axis). For instance, after a \code{Conv2D} layer with
\code{data_format="channels_first"}, set \code{axis=1} in \code{BatchNormalization}.}

\item{momentum}{Momentum for the moving average.}

\item{epsilon}{Small float added to variance to avoid dividing by zero.}

\item{center}{If \code{TRUE}, add offset of \code{beta} to normalized tensor. If \code{FALSE},
\code{beta} is ignored.}

\item{scale}{If \code{TRUE}, multiply by \code{gamma}. If \code{FALSE}, \code{gamma} is not used. When
the next layer is linear (also e.g. \code{nn.relu}), this can be disabled
since the scaling will be done by the next layer.}

\item{beta_initializer}{Initializer for the beta weight.}

\item{gamma_initializer}{Initializer for the gamma weight.}

\item{moving_mean_initializer}{Initializer for the moving mean.}

\item{moving_variance_initializer}{Initializer for the moving variance.}

\item{beta_regularizer}{Optional regularizer for the beta weight.}

\item{gamma_regularizer}{Optional regularizer for the gamma weight.}

\item{beta_constraint}{Optional constraint for the beta weight.}

\item{gamma_constraint}{Optional constraint for the gamma weight.}

\item{synchronized}{If \code{TRUE}, synchronizes the global batch statistics (mean and
variance) for the layer across all devices at each training step in a
distributed training strategy. If \code{FALSE}, each replica uses its own
local batch statistics. Only relevant when used inside a
\code{tf$distribute} strategy.}

\item{...}{standard layer arguments.}
}
\description{
Layer that normalizes its inputs
}
\details{
Batch normalization applies a transformation that maintains the mean output
close to 0 and the output standard deviation close to 1.

Importantly, batch normalization works differently during training and
during inference.

\strong{During training} (i.e. when using \code{fit()} or when calling the layer/model
with the argument \code{training=TRUE}), the layer normalizes its output using
the mean and standard deviation of the current batch of inputs. That is to
say, for each channel being normalized, the layer returns
\code{gamma * (batch - mean(batch)) / sqrt(var(batch) + epsilon) + beta}, where:
\itemize{
\item \code{epsilon} is small constant (configurable as part of the constructor
arguments)
\item \code{gamma} is a learned scaling factor (initialized as 1), which
can be disabled by passing \code{scale=FALSE} to the constructor.
\item \code{beta} is a learned offset factor (initialized as 0), which
can be disabled by passing \code{center=FALSE} to the constructor.
}

\strong{During inference} (i.e. when using \code{evaluate()} or \code{predict()} or when
calling the layer/model with the argument \code{training=FALSE} (which is the
default), the layer normalizes its output using a moving average of the
mean and standard deviation of the batches it has seen during training. That
is to say, it returns
\code{gamma * (batch - self.moving_mean) / sqrt(self.moving_var+epsilon) + beta}.

\code{self$moving_mean} and \code{self$moving_var} are non-trainable variables that
are updated each time the layer in called in training mode, as such:
\itemize{
\item \code{moving_mean = moving_mean * momentum + mean(batch) * (1 - momentum)}
\item \code{moving_var = moving_var * momentum + var(batch) * (1 - momentum)}
}

As such, the layer will only normalize its inputs during inference
\emph{after having been trained on data that has similar statistics as the
inference data}.

When \code{synchronized=TRUE} is set and if this layer is used within a
\code{tf$distribute} strategy, there will be an \code{allreduce} call
to aggregate batch statistics across all replicas at every
training step. Setting \code{synchronized} has no impact when the model is
trained without specifying any distribution strategy.

Example usage:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{strategy <- tf$distribute$MirroredStrategy()

with(strategy$scope(), \{
  model <- keras_model_sequential()
  model \%>\%
    layer_dense(16) \%>\%
    layer_batch_normalization(synchronized=TRUE)
\})
}\if{html}{\out{</div>}}
}
\seealso{
\itemize{
\item \url{https://www.tensorflow.org/api_docs/python/tf/keras/layers/BatchNormalization}
\item \url{https://keras.io/api/layers}
}
}
