% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ifit.R
\name{ifit}
\alias{ifit}
\title{Simulation-based (indirect) fitting of parametric models}
\usage{
ifit(
  tobs,
  tsim,
  l,
  u,
  cluster = NULL,
  export = NULL,
  trace = 0,
  Ntotal = 1e+06,
  NTotglobal = 20000,
  Ninit = 1000,
  Nelite = 100,
  Aelite = 0.5,
  Tolglobal = 0.1,
  Tollocal = 1,
  Tolmodel = 1.5,
  NFitlocal = 4000,
  NAddglobal = 100,
  NAddlocal = 10,
  Lambda = 0.1,
  Rhomax = 0.1
)
}
\arguments{
\item{tobs}{A vector containing the observed summary statistics.}

\item{tsim}{A function implementing the model to be simulated. It
must take as arguments a vector of model parameter values and
it must return a vector of summary statistics.}

\item{l, u}{Two numeric vectors of length equal to the number of the
parameters containing the lower and upper bounds on the parameters.}

\item{cluster}{if not null, the model will be simulated in parallel;
the argument can be either a cluster object (as defined in package
\code{parallel}) or a positive integer (in this case the cluster will be created and
deleted at the end by \code{ifit}).}

\item{export}{a vector giving the names of the global objects
(tipically needed by function \code{tsim}) which should be exported to the
cluster; not used if \code{cluster} is null.}

\item{trace}{An integer value; if greater than zero, results will be
printed (approximately) every \code{trace} model simulations.}

\item{Ntotal}{The maximum number of allowed model simulations.}

\item{NTotglobal}{The maximum number of simulations performed during
the global search phase.}

\item{Ninit, Nelite, Aelite, Tolglobal, Tollocal, Tolmodel, NFitlocal, NAddglobal, NAddlocal, Rhomax, Lambda}{Constants affecting the details of the algorithm (see the vignette
describing the algorithm)}
}
\value{
An object of class \code{ifit} for which a suitable \code{print} method
exists and whose elements can be accessed using the
functions described in \link{ifit-methods}
}
\description{
Fits an untractable parametric model by simulation
}
\examples{
# A toy model
set.seed(1)
n <- 3
y <- rnorm(n, 1)
tobs <- mean(y)
tsim <- function(theta) mean(rnorm(n, theta[1]))
m <- ifit(tobs, tsim, l = -3, u = 3)
m
coef(m)
confint(m)
globalIFIT(m)
numsimIFIT(m)
vcov(m, "parameters")
vcov(m, "statistics")
jacobianIFIT(m)
diagIFIT(m, plot = FALSE)
estfunIFIT(m)

\donttest{
# Logit model 
# It takes some time to run
n <- 100
X <- seq(-1, 1, length = n)
Z <- rnorm(n)
X <- cbind(1, X, Z, Z + rnorm(n))
logitSim <- function(theta) rbinom(n, 1, 1 / (1 + exp(-X \%*\% theta)))
logitStat <- function(y) as.numeric(crossprod(X, y))
theta <- c(-1, 1, 0.5, -0.5)
y <- logitSim(theta)
m <- ifit(
    tobs = logitStat(y), tsim = function(t) logitStat(logitSim(t)),
    l = rep(-5, 4), u = rep(5, 4), trace = 1000
)
m
g <- glm(y ~ X - 1, family = binomial)
rbind(
    ifit.estimates = coef(m),
    glm.estimates = coef(g),
    ifit.se = sqrt(diag(vcov(m))),
    glm.se = sqrt(diag(vcov(g)))
)
}
}
\references{
Guido Masarotto (2015) 'Simulation-Based Fitting of Intractable
Models  via Sequential Sampling and Local Smoothing',
arXiv eprint 2511.08180, pp. 1-23, \doi{10.48550/arXiv.2511.08180}
}
\keyword{inference}
