/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HGmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HGpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

#define H5HG_MAXLINK 65535

#define H5HG_MAXIDX 65535

static haddr_t H5HG__create(H5F_t *f, size_t size);
static size_t  H5HG__alloc(H5F_t *f, H5HG_heap_t *heap, size_t size, unsigned *heap_flags_ptr);

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE(H5HG_heap_t);

H5FL_SEQ_DEFINE(H5HG_obj_t);

H5FL_BLK_DEFINE(gheap_chunk);

static haddr_t
H5HG__create(H5F_t *f, size_t size)
{
    H5HG_heap_t *heap = NULL;
    uint8_t     *p    = NULL;
    haddr_t      addr = HADDR_UNDEF;
    size_t       n;
    haddr_t      ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    if (size < H5HG_MINSIZE)
        size = H5HG_MINSIZE;
    size = H5HG_ALIGN(size);

    
    H5_CHECK_OVERFLOW(size, size_t, hsize_t);
    if (HADDR_UNDEF == (addr = H5MF_alloc(f, H5FD_MEM_GHEAP, (hsize_t)size)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF, "unable to allocate file space for global heap");
    if (NULL == (heap = H5FL_CALLOC(H5HG_heap_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, HADDR_UNDEF, "memory allocation failed");
    heap->addr   = addr;
    heap->size   = size;
    heap->shared = H5F_SHARED(f);

    if (NULL == (heap->chunk = H5FL_BLK_MALLOC(gheap_chunk, size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, HADDR_UNDEF, "memory allocation failed");
    memset(heap->chunk, 0, size);
    heap->nalloc = H5HG_NOBJS(f, size);
    heap->nused  = 1; 
    if (NULL == (heap->obj = H5FL_SEQ_MALLOC(H5HG_obj_t, heap->nalloc)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, HADDR_UNDEF, "memory allocation failed");

    
    H5MM_memcpy(heap->chunk, H5HG_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    p    = heap->chunk + H5_SIZEOF_MAGIC;
    *p++ = H5HG_VERSION;
    *p++ = 0; 
    *p++ = 0; 
    *p++ = 0; 
    H5F_ENCODE_LENGTH(f, p, size);

    
    n = (size_t)H5HG_ALIGN(p - heap->chunk) - (size_t)(p - heap->chunk);
    p += n;

    
    heap->obj[0].size = size - H5HG_SIZEOF_HDR(f);
    assert(H5HG_ISALIGNED(heap->obj[0].size));
    heap->obj[0].nrefs = 0;
    heap->obj[0].begin = p;
    UINT16ENCODE(p, 0); 
    UINT16ENCODE(p, 0); 
    UINT32ENCODE(p, 0); 
    H5F_ENCODE_LENGTH(f, p, heap->obj[0].size);

    
    if (H5F_cwfs_add(f, heap) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF,
                    "unable to add global heap collection to file's CWFS");

    
    if (H5AC_insert_entry(f, H5AC_GHEAP, addr, heap, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF, "unable to cache global heap collection");

    ret_value = addr;

done:
    
    if (!H5_addr_defined(ret_value)) {
        if (H5_addr_defined(addr)) {
            
            if (H5MF_xfree(f, H5FD_MEM_GHEAP, addr, (hsize_t)size) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, HADDR_UNDEF, "unable to free global heap");

            
            if (heap)
                
                if (H5HG__free(heap) < 0)
                    HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, HADDR_UNDEF,
                                "unable to destroy global heap collection");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HG_heap_t *
H5HG__protect(H5F_t *f, haddr_t addr, unsigned flags)
{
    H5HG_heap_t *heap;             
    H5HG_heap_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    if (NULL == (heap = (H5HG_heap_t *)H5AC_protect(f, H5AC_GHEAP, addr, f, flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect global heap");

    
    heap->addr = addr;

    
    ret_value = heap;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5HG__alloc(H5F_t *f, H5HG_heap_t *heap, size_t size, unsigned *heap_flags_ptr)
{
    size_t   idx;
    uint8_t *p;
    size_t   need      = H5HG_SIZEOF_OBJHDR(f) + H5HG_ALIGN(size);
    size_t   ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(heap);
    assert(heap->obj[0].size >= need);
    assert(heap_flags_ptr);

    
    if (heap->nused <= H5HG_MAXIDX)
        idx = heap->nused++;
    else {
        for (idx = 1; idx < heap->nused; idx++)
            if (NULL == heap->obj[idx].begin)
                break;
    } 

    assert(idx < heap->nused);

    
    if (idx >= heap->nalloc) {
        size_t      new_alloc; 
        H5HG_obj_t *new_obj;   

        
        
        new_alloc = MIN(MAX(heap->nalloc * 2, (idx + 1)), (H5HG_MAXIDX + 1));
        assert(idx < new_alloc);

        
        if (NULL == (new_obj = H5FL_SEQ_REALLOC(H5HG_obj_t, heap->obj, new_alloc)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, 0, "memory allocation failed");

        
        memset(&new_obj[heap->nalloc], 0, (new_alloc - heap->nalloc) * sizeof(heap->obj[0]));

        
        heap->nalloc = new_alloc;
        heap->obj    = new_obj;
        assert(heap->nalloc > heap->nused);
    } 

    
    heap->obj[idx].nrefs = 0;
    heap->obj[idx].size  = size;
    heap->obj[idx].begin = heap->obj[0].begin;
    p                    = heap->obj[idx].begin;
    UINT16ENCODE(p, idx);
    UINT16ENCODE(p, 0); 
    UINT32ENCODE(p, 0); 
    H5F_ENCODE_LENGTH(f, p, size);

    
    if (need == heap->obj[0].size) {
        
        heap->obj[0].size  = 0;
        heap->obj[0].begin = NULL;
    } 
    else if (heap->obj[0].size - need >= H5HG_SIZEOF_OBJHDR(f)) {
        
        heap->obj[0].size -= need;
        heap->obj[0].begin += need;
        p = heap->obj[0].begin;
        UINT16ENCODE(p, 0); 
        UINT16ENCODE(p, 0); 
        UINT32ENCODE(p, 0); 
        H5F_ENCODE_LENGTH(f, p, heap->obj[0].size);
        assert(H5HG_ISALIGNED(heap->obj[0].size));
    } 
    else {
        
        heap->obj[0].size -= need;
        heap->obj[0].begin += need;
        assert(H5HG_ISALIGNED(heap->obj[0].size));
    }

    
    *heap_flags_ptr |= H5AC__DIRTIED_FLAG;

    
    ret_value = idx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HG_extend(H5F_t *f, haddr_t addr, size_t need)
{
    H5HG_heap_t *heap       = NULL;               
    unsigned     heap_flags = H5AC__NO_FLAGS_SET; 
    size_t       old_size;                        
    uint8_t     *new_chunk;                       
    uint8_t     *p;                               
    unsigned     u;                               
    herr_t       ret_value = SUCCEED;             

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    if (NULL == (heap = H5HG__protect(f, addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect global heap");

    
    if (NULL == (new_chunk = H5FL_BLK_REALLOC(gheap_chunk, heap->chunk, (heap->size + need))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "new heap allocation failed");
    memset(new_chunk + heap->size, 0, need);

    
    old_size = heap->size;
    heap->size += need;

    
    p = new_chunk + H5_SIZEOF_MAGIC + 1  + 3 ;
    H5F_ENCODE_LENGTH(f, p, heap->size);

    
    for (u = 0; u < heap->nused; u++)
        if (heap->obj[u].begin)
            heap->obj[u].begin = new_chunk + (heap->obj[u].begin - heap->chunk);

    
    heap->chunk = new_chunk;

    
    heap->obj[0].size += need;
    if (heap->obj[0].begin == NULL)
        heap->obj[0].begin = heap->chunk + old_size;
    p = heap->obj[0].begin;
    UINT16ENCODE(p, 0); 
    UINT16ENCODE(p, 0); 
    UINT32ENCODE(p, 0); 
    H5F_ENCODE_LENGTH(f, p, heap->obj[0].size);
    assert(H5HG_ISALIGNED(heap->obj[0].size));

    
    if (H5AC_resize_entry(heap, heap->size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize global heap in cache");

    
    heap_flags |= H5AC__DIRTIED_FLAG;

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, heap->addr, heap, heap_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to unprotect heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HG_insert(H5F_t *f, size_t size, const void *obj, H5HG_t *hobj )
{
    size_t       need; 
    size_t       idx;
    haddr_t      addr; 
    H5HG_heap_t *heap       = NULL;
    unsigned     heap_flags = H5AC__NO_FLAGS_SET;
    herr_t       ret_value  = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(H5AC__GLOBALHEAP_TAG, FAIL)

    
    assert(f);
    assert(0 == size || obj);
    assert(hobj);

    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    need = H5HG_SIZEOF_OBJHDR(f) + H5HG_ALIGN(size);

    
    addr = HADDR_UNDEF;
    if (H5F_cwfs_find_free_heap(f, need, &addr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "error trying to locate heap");

    
    if (!H5_addr_defined(addr)) {
        addr = H5HG__create(f, need + H5HG_SIZEOF_HDR(f));

        if (!H5_addr_defined(addr))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "unable to allocate a global heap collection");
    } 
    assert(H5_addr_defined(addr));

    if (NULL == (heap = H5HG__protect(f, addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect global heap");

    
    if (0 == (idx = H5HG__alloc(f, heap, size, &heap_flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "unable to allocate global heap object");

    
    if (size > 0)
        H5MM_memcpy(heap->obj[idx].begin + H5HG_SIZEOF_OBJHDR(f), obj, size);
    heap_flags |= H5AC__DIRTIED_FLAG;

    
    hobj->addr = heap->addr;
    hobj->idx  = idx;

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, heap->addr, heap, heap_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to unprotect heap.");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

void *
H5HG_read(H5F_t *f, H5HG_t *hobj, void *object , size_t *buf_size)
{
    H5HG_heap_t *heap = NULL;          
    size_t       size;                 
    uint8_t     *p;                    
    void        *orig_object = object; 
    void        *ret_value   = NULL;   

    FUNC_ENTER_NOAPI_TAG(H5AC__GLOBALHEAP_TAG, NULL)

    
    assert(f);
    assert(hobj);

    
    if (0 == hobj->idx)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    if (NULL == (heap = H5HG__protect(f, hobj->addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect global heap");
    if (hobj->idx >= heap->nused)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);
    if (NULL == heap->obj[hobj->idx].begin)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "bad heap pointer, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    size = heap->obj[hobj->idx].size;
    p    = heap->obj[hobj->idx].begin + H5HG_SIZEOF_OBJHDR(f);

    
    if (!object && NULL == (object = H5MM_malloc(size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    H5MM_memcpy(object, p, size);

    
    if (heap->obj[0].begin) {
        if (H5F_cwfs_advance_heap(f, heap, false) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTMODIFY, NULL, "can't adjust file's CWFS");
    } 

    
    if (buf_size)
        *buf_size = size;

    
    ret_value = object;

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, hobj->addr, heap, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, NULL, "unable to release object header");

    if (NULL == ret_value && NULL == orig_object && object)
        H5MM_free(object);

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

int
H5HG_link(H5F_t *f, const H5HG_t *hobj, int adjust)
{
    H5HG_heap_t *heap       = NULL;
    unsigned     heap_flags = H5AC__NO_FLAGS_SET;
    int          ret_value  = -1; 

    FUNC_ENTER_NOAPI_TAG(H5AC__GLOBALHEAP_TAG, FAIL)

    
    assert(f);
    assert(hobj);
    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    if (0 == hobj->idx)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    if (NULL == (heap = H5HG__protect(f, hobj->addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect global heap");

    if (adjust != 0) {
        if (hobj->idx >= heap->nused)
            HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                        hobj->addr, (unsigned long long)hobj->idx);
        if (NULL == heap->obj[hobj->idx].begin)
            HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap pointer, heap object = {%" PRIxHADDR ", %llu}",
                        hobj->addr, (unsigned long long)hobj->idx);
        if ((heap->obj[hobj->idx].nrefs + adjust) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, FAIL, "new link count would be out of range");
        if ((heap->obj[hobj->idx].nrefs + adjust) > H5HG_MAXLINK)
            HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "new link count would be out of range");
        heap->obj[hobj->idx].nrefs += adjust;
        heap_flags |= H5AC__DIRTIED_FLAG;
    } 

    
    ret_value = heap->obj[hobj->idx].nrefs;

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, hobj->addr, heap, heap_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5HG_get_obj_size(H5F_t *f, H5HG_t *hobj, size_t *obj_size)
{
    H5HG_heap_t *heap      = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(H5AC__GLOBALHEAP_TAG, FAIL)

    
    assert(f);
    assert(hobj);
    assert(obj_size);

    
    if (0 == hobj->idx)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    if (NULL == (heap = H5HG__protect(f, hobj->addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect global heap");

    
    if (hobj->idx >= heap->nused)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);
    if (NULL == heap->obj[hobj->idx].begin)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap pointer, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    *obj_size = heap->obj[hobj->idx].size;

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, hobj->addr, heap, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5HG_remove(H5F_t *f, H5HG_t *hobj)
{
    H5HG_heap_t *heap = NULL;
    uint8_t     *p = NULL, *obj_start = NULL;
    size_t       need;
    unsigned     u;
    unsigned     flags     = H5AC__NO_FLAGS_SET; 
    herr_t       ret_value = SUCCEED;            

    FUNC_ENTER_NOAPI_TAG(H5AC__GLOBALHEAP_TAG, FAIL)

    
    assert(f);
    assert(hobj);
    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    if (0 == hobj->idx)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    if (NULL == (heap = H5HG__protect(f, hobj->addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect global heap");

    
    if (hobj->idx >= heap->nused)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap index, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    
    if (heap->obj[hobj->idx].nrefs == 0 && heap->obj[hobj->idx].size == 0 && !heap->obj[hobj->idx].begin)
        HGOTO_DONE(SUCCEED);

    
    if (NULL == heap->obj[hobj->idx].begin)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad heap pointer, heap object = {%" PRIxHADDR ", %llu}",
                    hobj->addr, (unsigned long long)hobj->idx);

    obj_start = heap->obj[hobj->idx].begin;
    
    need = H5HG_ALIGN(heap->obj[hobj->idx].size) + H5HG_SIZEOF_OBJHDR(f);

    
    for (u = 0; u < heap->nused; u++)
        if (heap->obj[u].begin > heap->obj[hobj->idx].begin)
            heap->obj[u].begin -= need;
    if (NULL == heap->obj[0].begin) {
        heap->obj[0].begin = heap->chunk + (heap->size - need);
        heap->obj[0].size  = need;
        heap->obj[0].nrefs = 0;
    } 
    else
        heap->obj[0].size += need;
    memmove(obj_start, obj_start + need, heap->size - (size_t)((obj_start + need) - heap->chunk));
    if (heap->obj[0].size >= H5HG_SIZEOF_OBJHDR(f)) {
        p = heap->obj[0].begin;
        UINT16ENCODE(p, 0); 
        UINT16ENCODE(p, 0); 
        UINT32ENCODE(p, 0); 
        H5F_ENCODE_LENGTH(f, p, heap->obj[0].size);
    } 
    memset(heap->obj + hobj->idx, 0, sizeof(H5HG_obj_t));
    flags |= H5AC__DIRTIED_FLAG;

    if ((heap->obj[0].size + H5HG_SIZEOF_HDR(f)) == heap->size) {
        
        flags |=
            H5AC__DELETED_FLAG |
            H5AC__FREE_FILE_SPACE_FLAG; 
    }                                   
    else {
        
        if (H5F_cwfs_advance_heap(f, heap, true) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTMODIFY, FAIL, "can't adjust file's CWFS");
    } 

done:
    if (heap && H5AC_unprotect(f, H5AC_GHEAP, hobj->addr, heap, flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5HG__free(H5HG_heap_t *heap)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(heap);

    
    if (H5F_cwfs_remove_heap(heap->shared, heap) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove heap from file's CWFS");

    if (heap->chunk)
        heap->chunk = H5FL_BLK_FREE(gheap_chunk, heap->chunk);
    if (heap->obj)
        heap->obj = H5FL_SEQ_FREE(H5HG_obj_t, heap->obj);
    heap = H5FL_FREE(H5HG_heap_t, heap);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
