% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_design_wlr.R
\name{gs_design_wlr}
\alias{gs_design_wlr}
\title{Group sequential design using weighted log-rank test under non-proportional hazards}
\usage{
gs_design_wlr(
  enroll_rate = define_enroll_rate(duration = c(2, 2, 10), rate = c(3, 6, 9)),
  fail_rate = tibble(stratum = "All", duration = c(3, 100), fail_rate = log(2)/c(9, 18),
    hr = c(0.9, 0.6), dropout_rate = rep(0.001, 2)),
  weight = "logrank",
  approx = "asymptotic",
  alpha = 0.025,
  beta = 0.1,
  ratio = 1,
  info_frac = NULL,
  info_scale = c("h0_h1_info", "h0_info", "h1_info"),
  analysis_time = 36,
  binding = FALSE,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = alpha),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = beta),
  test_upper = TRUE,
  test_lower = TRUE,
  h1_spending = TRUE,
  r = 18,
  tol = 1e-06,
  interval = c(0.01, 1000)
)
}
\arguments{
\item{enroll_rate}{An \code{enroll_rate} data frame with or without stratum
created by \code{\link[=define_enroll_rate]{define_enroll_rate()}}.}

\item{fail_rate}{A \code{fail_rate} data frame with or without stratum
created by \code{\link[=define_fail_rate]{define_fail_rate()}}.}

\item{weight}{Weight of weighted log rank test:
\itemize{
\item \code{"logrank"} = regular logrank test.
\item \code{list(method = "fh", param = list(rho = ..., gamma = ...))} = Fleming-Harrington weighting functions.
\item \code{list(method = "mb", param = list(tau = ..., w_max = ...))} = Magirr and Burman weighting functions.
}}

\item{approx}{Approximate estimation method for Z statistics.
\itemize{
\item \code{"event_driven"} = only work under proportional hazard model with log rank test.
\item \code{"asymptotic"}.
}}

\item{alpha}{One-sided Type I error.}

\item{beta}{Type II error.}

\item{ratio}{Experimental:Control randomization ratio.}

\item{info_frac}{Targeted information fraction for analyses. See details.}

\item{info_scale}{Information scale for calculation. Options are:
\itemize{
\item \code{"h0_h1_info"} (default): variance under both null and alternative hypotheses is used.
\item \code{"h0_info"}: variance under null hypothesis is used.
This is often used for testing methods that use local alternatives, such as the Schoenfeld method.
\item \code{"h1_info"}: variance under alternative hypothesis is used.
}}

\item{analysis_time}{Targeted calendar timing of analyses. See details.}

\item{binding}{Indicator of whether futility bound is binding;
default of \code{FALSE} is recommended.}

\item{upper}{Function to compute upper bound.
\itemize{
\item \code{gs_spending_bound()}: alpha-spending efficacy bounds.
\item \code{gs_b()}: fixed efficacy bounds.
}}

\item{upar}{Parameters passed to \code{upper}.
\itemize{
\item If \code{upper = gs_b}, then \code{upar} is a numerical vector specifying the fixed efficacy bounds per analysis.
\item If \code{upper = gs_spending_bound}, then \code{upar} is a list including
\itemize{
\item \code{sf} for the spending function family.
\item \code{total_spend} for total alpha spend.
\item \code{param} for the parameter of the spending function.
\item \code{timing} specifies spending time if different from information-based spending; see details.
}
}}

\item{lower}{Function to compute lower bound, which can be set up similarly as \code{upper}.
See \href{https://merck.github.io/gsDesign2/articles/story-seven-test-types.html}{this vignette}.}

\item{lpar}{Parameters passed to \code{lower}, which can be set up similarly as \code{upar.}}

\item{test_upper}{Indicator of which analyses should include
an upper (efficacy) bound;
single value of \code{TRUE} (default) indicates all analyses; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have an efficacy bound.}

\item{test_lower}{Indicator of which analyses should include a lower bound;
single value of \code{TRUE} (default) indicates all analyses;
single value of \code{FALSE} indicated no lower bound; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have a lower bound.}

\item{h1_spending}{Indicator that lower bound to be set by spending
under alternate hypothesis (input \code{fail_rate})
if spending is used for lower bound.
If this is \code{FALSE}, then the lower bound spending is under the null hypothesis.
This is for two-sided symmetric or asymmetric testing under the null hypothesis;
See \href{https://merck.github.io/gsDesign2/articles/story-seven-test-types.html}{this vignette}.}

\item{r}{Integer value controlling grid for numerical integration as in
Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally, \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for boundary convergence (on Z-scale); normally not changed by the user.}

\item{interval}{An interval presumed to include the times at which
expected event count is equal to targeted event.
Normally, this can be ignored by the user as it is set to \code{c(.01, 1000)}.}
}
\value{
A list with input parameters, enrollment rate, analysis, and bound.
}
\description{
Group sequential design using weighted log-rank test under non-proportional hazards
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Validate if input analysis_time is a positive number or a positive increasing sequence.
   \item Validate if input info_frac is a positive number
   or positive increasing sequence on (0, 1] with final value of 1.
   \item Validate if inputs info_frac and analysis_time  have the same length if both have length > 1.
   \item Compute information at input analysis_time using \code{gs_info_wlr()}.
   \item Compute sample size and bounds using \code{gs_design_npe()}.
   \item Return a list of design enrollment, failure rates, and bounds.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
library(mvtnorm)
library(gsDesign)
library(gsDesign2)

# set enrollment rates
enroll_rate <- define_enroll_rate(duration = 12, rate = 1)

# set failure rates
fail_rate <- define_fail_rate(
  duration = c(4, 100),
  fail_rate = log(2) / 15, # median survival 15 month
  hr = c(1, .6),
  dropout_rate = 0.001
)

# Example 1 ----
# Information fraction driven design
gs_design_wlr(
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  ratio = 1,
  alpha = 0.025, beta = 0.2,
  weight = list(method = "mb", param = list(tau = Inf, w_max = 2)),
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.2),
  analysis_time = 36,
  info_frac = c(0.6, 1)
)

# Example 2 ----
# Calendar time driven design
gs_design_wlr(
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  ratio = 1,
  alpha = 0.025, beta = 0.2,
  weight = list(method = "mb", param = list(tau = Inf, w_max = 2)),
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.2),
  analysis_time = c(24, 36),
  info_frac = NULL
)

# Example 3 ----
# Both calendar time and information fraction driven design
gs_design_wlr(
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  ratio = 1,
  alpha = 0.025, beta = 0.2,
  weight = list(method = "mb", param = list(tau = Inf, w_max = 2)),
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.2),
  analysis_time = c(24, 36),
  info_frac = c(0.6, 1)
)
}
