% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-peaks.R
\name{stat_peaks}
\alias{stat_peaks}
\alias{stat_valleys}
\title{Local maxima (peaks) or minima (valleys)}
\usage{
stat_peaks(
  mapping = NULL,
  data = NULL,
  geom = "point",
  position = "identity",
  ...,
  span = 5,
  global.threshold = 0,
  local.threshold = 0,
  local.reference = "median",
  strict = FALSE,
  label.fmt = NULL,
  x.label.fmt = NULL,
  y.label.fmt = NULL,
  extract.peaks = NULL,
  orientation = "x",
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = TRUE
)

stat_valleys(
  mapping = NULL,
  data = NULL,
  geom = "point",
  position = "identity",
  ...,
  span = 5,
  global.threshold = 0.01,
  local.threshold = NULL,
  local.reference = "median",
  strict = FALSE,
  label.fmt = NULL,
  x.label.fmt = NULL,
  y.label.fmt = NULL,
  extract.valleys = NULL,
  orientation = "x",
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs to be
set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific dataset - only needed if you want to override
the plot defaults.}

\item{geom}{The geometric object to use display the data}

\item{position}{The position adjustment to use for overlapping points on this
layer}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}

\item{span}{odd positive integer A peak is defined as an element in a
sequence which is greater than all other elements within a moving window of
width \code{span} centred at that element. The default value is 5, meaning
that a peak is taller than its four nearest neighbours. \code{span = NULL}
extends the span to the whole length of \code{x}.}

\item{global.threshold}{numeric A value belonging to class \code{"AsIs"} is
interpreted as an absolute minimum height or depth expressed in data units.
A bare \code{numeric} value (normally between 0.0 and 1.0), is interpreted
as relative to \code{threshold.range}. In both cases it sets a
\emph{global} height (depth) threshold below which peaks (valleys) are
ignored. A bare negative \code{numeric} value indicates the \emph{global}
height (depth) threshold below which peaks (valleys) are be ignored. If
\code{global.threshold = NULL}, no threshold is applied and all peaks
returned.}

\item{local.threshold}{numeric A value belonging to class \code{"AsIs"} is
interpreted as an absolute minimum height (depth) expressed in data units
relative to a within-window computed reference value. A bare \code{numeric}
value (normally between 0.0 and 1.0), is interpreted as expressed in units
relative to \code{threshold.range}. In both cases \code{local.threshold}
sets a \emph{local} height (depth) threshold below which peaks (valleys)
are ignored. If \code{local.threshold = NULL} or if \code{span} spans the
whole of \code{x}, no threshold is applied.}

\item{local.reference}{character One of \code{"median"}, \code{"median.log"},
\code{"median.sqrt"}, \code{"farthest"}, \code{"farthest.log"} or
\code{"farthest.sqrt"}. The reference used to assess the height of the
peak, is either the minimum/maximum value within the window or the median
of all values in the window.}

\item{strict}{logical flag: if \code{TRUE}, an element must be strictly
greater than all other values in its window to be considered a peak.
Default: \code{FALSE} (since version 0.13.1).}

\item{label.fmt, x.label.fmt, y.label.fmt}{character  strings giving a format
definition for construction of character strings labels with function
\code{\link{sprintf}} from \code{x} and/or \code{y} values.}

\item{extract.peaks, extract.valleys}{If \code{TRUE} only the rows containing
peaks or valleys are returned. If \code{FALSE} the whole of \code{data} is
returned but with labels set to \code{NA} in rows not containing peaks or
valleys. If \code{NULL}, the default, \code{TRUE}, is used unless the geom
name passed as argument is \code{"text_repel"} or \code{"label_repel"}.}

\item{orientation}{character The orientation of the layer can be set to
either \code{"x"}, the default, or \code{"y"}.}

\item{na.rm}{a logical value indicating whether NA values should be stripped
before the computation proceeds.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and shouldn't inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}
}
\value{
A data frame with one row for each peak (or valley) found in the data
  extracted from the input \code{data} or all rows in data. Added columns
  contain the labels.
}
\description{
\code{stat_peaks} finds at which x positions the global y maximun or local y
maxima are located. \code{stat_valleys} finds at which x positions the global
y minimum or local y minima located. They both support filtering of relevant
peaks. \strong{Axis flipping is supported.}
}
\details{
As \code{find_valleys}, \code{stat_peaks} and \code{stat_valleys}
  call \code{find_peaks} to search for peaks or valleys, this description
  applies to all four functions.

  Function \code{find_peaks} is a wrapper built onto function
  \code{\link[splus2R]{peaks}} from \pkg{splus2R}, adds support for peak
  height thresholds and handles \code{span = NULL} and non-finite (including
  NA) values differently than \code{splus2R::peaks}. Instead of giving an
  error when \code{na.rm = FALSE} and \code{x} contains \code{NA} values,
  \code{NA} values are replaced with the smallest finite value in \code{x}.
  \code{span = NULL} is treated as a special case and selects \code{max(x)}.
  Passing `strict = TRUE` ensures that multiple global and within window
  maxima are ignored, and can result in no peaks being returned.#'

  Two tests make it possible to ignore irrelevant peaks. One test
  (\code{global.threshold}) is based on the absolute height of the peaks and
  can be used in all cases to ignore globally low peaks. A second test
  (\code{local.threshold}) is available when the window defined by `span`
  does not include all observations and can be used to ignore peaks that are
  not locally prominent. In this second approach the height of each peak is
  compared to a summary computed from other values within the window of width
  equal to \code{span} where it was found. In this second case, the reference
  value used within each window containing a peak is given by
  \code{local.reference}. Parameter \code{threshold.range} determines how the
  bare \code{numeric} values passed as argument to \code{global.threshold}
  and \code{local.threshold} are scaled. The default, \code{NULL} uses the
  range of \code{x}. Thresholds for ignoring too small peaks are applied
  after peaks are searched for, and threshold values can in some cases result
  in no peaks being found. If either threshold is not available (\code{NA})
  the returned value is a \code{NA} vector of the same length as \code{x}.

  The \code{local.threshold} argument is used \emph{as is} when
  \code{local.reference} is \code{"median"} or \code{"farthest"}, i.e., the
  same distance between peak and reference is used as cut-off irrespective of
  the value of the reference. In cases when the prominence of peaks is
  positively correlated with the baseline, a \code{local.threshold} that
  increases together with increasing computed within window median or
  farthest value applies apply a less stringent height requirement in regions
  with overall low height. In this case, natural logarithm or square root
  weighting can be requested with `local.reference` arguments `"median.log"`,
  `"farthest.log"`, `"median.sqrt"`, and `"farthest.sqrt"` as arguments for
  \code{local.reference}.
}
\note{
\code{stat_peaks} and \code{stat_valleys} work nicely together with
  geoms \code{geom_text_repel} and \code{geom_label_repel} from package
  \code{\link[ggrepel]{ggrepel}} to solve the problem of overlapping labels
  by displacing them. To discard overlapping labels use \code{check_overlap =
  TRUE} as argument to \code{geom_text}.

  By default the labels are character values ready to be added as is, but
  with a suitable \code{label.fmt} labels suitable for parsing by the geoms
  (e.g. into expressions containing Greek letters or super or subscripts) can
  be also easily obtained.
}
\section{Computed and copied variables in the returned data frame}{

\describe{
  \item{x}{x-value at the peak (or valley) as numeric}
  \item{y}{y-value at the peak (or valley) as numeric}
  \item{x.label}{x-value at the peak (or valley) formatted as character}
  \item{y.label}{y-value at the peak (or valley) formatted as character}
}
}

\section{Default aesthetics}{
 These stats use \code{geom_point} by default as
  it is the geom most likely to work well in almost any situation without.
  The default aesthetics set by these stats allow their direct use with
  \code{geom_text}, \code{geom_label}, \code{geom_line}, \code{geom_rug},
  \code{geom_hline} and \code{geom_vline}. The formatting of the labels
  returned can be controlled by the user.

  Default aesthetics mapped by the statistic and available to geoms, in
  addtion to the automatically set required aesthetics.
\describe{
  \item{label}{stat(x.label)}
  \item{xintercept}{stat(x)}
  \item{yintercept}{stat(y)}
}
}

\section{Required aesthetics}{
 Required by the statistic and need to be set
  with \code{aes()}. Date time scales are recognized and labels
  formatted accordingly.

\describe{
  \item{x}{numeric or date time, independent variable}
  \item{y}{numeric, response variable where peaks or valleys are searched}
}
}

\examples{
# lynx and Nile are time.series objects recognized by
# ggpp::ggplot.ts() and converted on-the-fly with a default mapping

# numeric, date times and dates are supported with data frames

# using defaults
ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red") +
  stat_valleys(colour = "blue")

# using wider window
ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red", span = 11) +
  stat_valleys(colour = "blue", span = 11)

# global threshold for peak height
ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red",
             global.threshold = 0.5) # half of data range

ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red",
             global.threshold = I(1100)) + # data unit
             expand_limits(y = c(0, 1500))

# local (within window) threshold for peak height
# narrow peaks at the tip and locally tall

ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red",
             span = 9,
             local.threshold = 0.3,
             local.reference = "farthest")

# with narrower window
ggplot(Nile) +
  geom_line() +
  stat_peaks(colour = "red",
             span = 5,
             local.threshold = 0.25,
             local.reference = "farthest")

ggplot(lynx) +
  geom_line() +
  stat_peaks(colour = "red",
             local.threshold = 1/5,
             local.reference = "median")

ggplot(Nile) +
  geom_line() +
  stat_valleys(colour = "blue",
               global.threshold = I(700))

# orientation is supported
ggplot(lynx, aes(lynx, time)) +
  geom_line(orientation = "y") +
  stat_peaks(colour = "red", orientation = "y") +
  stat_valleys(colour = "blue", orientation = "y")

# default aesthetic mapping supports additional geoms
ggplot(lynx) +
  geom_line() +
  stat_peaks(colour = "red") +
  stat_peaks(colour = "red", geom = "rug")

ggplot(lynx) +
  geom_line() +
  stat_peaks(colour = "red") +
  stat_peaks(colour = "red", geom = "text", hjust = -0.1, angle = 33)

ggplot(lynx, aes(lynx, time)) +
  geom_line(orientation = "y") +
  stat_peaks(colour = "red", orientation = "y") +
  stat_peaks(colour = "red", orientation = "y",
             geom = "text", hjust = -0.1)

# Force conversion of time series time into POSIXct date time
ggplot(lynx, as.numeric = FALSE) +
  geom_line() +
  stat_peaks(colour = "red") +
  stat_peaks(colour = "red",
             geom = "text",
             hjust = -0.1,
             x.label.fmt = "\%Y",
             angle = 33)

ggplot(Nile, as.numeric = FALSE) +
  geom_line() +
  stat_peaks(colour = "red") +
  stat_peaks(colour = "red",
             geom = "text_s",
             position = position_nudge_keep(x = 0, y = 60),
             hjust = -0.1,
             x.label.fmt = "\%Y",
             angle = 90) +
  expand_limits(y = 2000)

ggplot(lynx, as.numeric = FALSE) +
  geom_line() +
  stat_peaks(colour = "red",
             geom = "text_s",
             position = position_nudge_to(y = 7600),
             arrow = arrow(length = grid::unit(1.5, "mm")),
             point.padding = 0.7,
             x.label.fmt = "\%Y",
             angle = 90) +
  expand_limits(y = 9000)

}
\seealso{
\code{\link{find_peaks}}, which is used internally.

\code{\link{find_peaks}}, for the functions used to located the
  peaks and valleys.
}
