% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/enforce.r
\name{enforce}
\alias{enforce}
\title{Ensure the truth of R expressions and cast/recycle objects.}
\usage{
enforce(..., .env = caller_env(), .error_call = caller_env())
}
\arguments{
\item{...}{any number of R expressions or formulas to be evaluated.
Expressions must evaluate to logical whilst formulas can use \link{c} on
the lhs and either functions or formulas that evaluate to logical,
or one of the type/size functions: \link{cast}, \link{recycle} or
\link{coerce} on the rhs. The rhs of a formula can also be a \link{list}
of multiple functions/formulas/calls. If an expression is named, or
if the list element on the rhs of a formula is named, the name is passed to
\link[cli:format_inline]{format_inline} and is used in the error message.}

\item{.env}{the environment to use for the evaluation of the
expressions and the assignment of the objects.}

\item{.error_call}{the call environment to use for error messages
(passed to \link[rlang:abort]{abort}).}
}
\value{
NULL, but objects casted/recycled in \code{...} will be changed in
the \code{.env} environment specified.
}
\description{
If any of the expressions in \code{...} are not all \code{TRUE}, \link[rlang:abort]{abort}
is called for the first which was not (\link{all}) \code{TRUE}. Alternatively,
\href{https://rlang.r-lib.org/reference/new_formula.html}{rlang formulas} can
be used to pass multiple objects to validation formulas/functions,
and/or attempt safe type casting and size recycling  using the
\link{cast}, \link{recycle} and \link{coerce} functions. The rhs of
formulas can be given in a \link{list} to pass multiple functions/formulas/calls.
Expressions are evaluated in the environment specified and objects
are assigned back into that environment. Type casting and recycling
are undertaken using the \href{https://vctrs.r-lib.org/}{vctrs} package
and thus apply \href{https://vctrs.r-lib.org/articles/type-size.html}{vctrs type and size rules}.
}
\details{
See \link{abort_if_not} for only validations and
\link{schema} for a data-masked version of this function.
}
\examples{
# NB: Some of these examples are expected to produce an error. To
#     prevent them from terminating a run with example() they are
#     piped into a call to try().

x <- 1L
y <- "hi"
z <- \(x) x > 1
enforce(x == 1, is.character(y), is.function(z)) # all TRUE

enforce(x == 2) |> try()

# A custom error message can be given for each expression by
# naming it:
enforce(
  "{.var y} must be {.cls numeric}, check input" = is.numeric(y)
) |> try()

# Formulas can be used to take pass multiple objects
# on the lhs, with functions/additional formulas required on
# the rhs:
enforce(
  "multiple objects using: {.fn c}" = c(x, y) ~ is.integer
) |> try()

# Formulas can also be used with `cast()`, `recycle()`, and
# `coerce()` on the rhs to safely cast or recycle objects:
enforce(x ~ cast(double()))
class(x) # x is now numeric
enforce(x ~ recycle(5))
length(x) # x is now length 5
enforce(y ~ coerce(type = factor(), size = 5))
print(y) # y is now factor and length 5

# Multiple calls can be used with formulas by wrapping them
# in `list()`, with the names of list elements being
# preferentially chosen for error messaging and the error
# message also showing which formula/function/call caused the
# error:
enforce(
  "generic message" = c(x, y, z) ~ list(
    Negate(is.null),
    "{.var specific} message" = Negate(is.function)
  )
) |> try()

# Changed elements are available immediately:
x <- y <- 1L
enforce(x ~ cast(double()), y ~ cast(x))
cat(class(x), class(y)) # both now numeric

# The `.error_call` argument can be used to specify where the
# error occurs, by default this is the caller environment:
myfunc <- function(...) enforce(...)
myfunc(x > 4) |> try()

# rlang injection can be used:
msg <- "{.var injection} msg"
cols <- quote(c(x, y))
enforce(!!msg := !!cols ~ is.integer) |> try()

# Objects are reverted to their original values if an error
# occur:
x <- y <- 1L
enforce(
  x ~ cast(double()), y ~ recycle(5), y ~ is.function
) |> try() # errors
class(x) # integer
length(y) # 1
}
